(ns tech.verify.persist-compliance
  (:require [clojure.test :refer [testing is]]
            [tech.resource :as resource]
            [tech.persist :as persist])
  (:import [java.util UUID]))

(defn level-1
  [persister-fn]
  (testing "Compliance level 1, basic support for :assoc! mutations"
    (resource/stack-resource-context
     (persist/with-persister (persister-fn)
       (let [item0 (first (persist/assoc! {:resource/type :one} :a 1 :b 2))
             item1 (first (persist/assoc! item0 :a 3))]
         (is (= {:resource/type :one :a 1 :b 2} (select-keys item0 [:resource/type :a :b])))
         (is (= (:resource/id item0) (:resource/id item1)))
         (is (= (:a item1) 3)))))))

(defn level-2
  [persister-fn]
  (testing "Compliance level 2, basic transactions, with flying reads"
    (resource/stack-resource-context
     (persist/with-persister (persister-fn)
       (let [item0 (first (persist/assoc! {:resource/type :one} :a 1 :b 2))
             [next-item item1] (persist/with-transaction
                                 (persist/assoc! {:resource/type :one} :a 1 :b 2)
                                 (persist/assoc! item0 :a 4)
                                 ;;check live reads from the temp index
                                 (is (= 4 (get-in (persist/index) [(:resource/id item0) :a]))))
             ;;Order is not guaranteed
             [next-item item1] (if (= (:resource/id item0) (:resource/id item1))
                                 [next-item item1]
                                 [item1 next-item])]
         (is (= {:resource/type :one :a 1 :b 2} (select-keys next-item [:resource/type :a :b])))
         (is (= (:resource/id item0) (:resource/id item1)))
         (is (= {:resource/type :one :a 4 :b 2} (select-keys item1 [:resource/type :a :b]))))))))

(defn level-3
  [persister-fn]
  (testing "Compliance level 3, basic support for :dissoc! mutations"
    (resource/stack-resource-context
     (persist/with-persister (persister-fn)
       (let [[item0] (persist/assoc! {:resource/type :one} :a 1 :b 2)
             [item1] (persist/dissoc! item0 :b)]
         (is (= {:resource/type :one :a 1} (select-keys item1 [:resource/type :a :b]))))))))

(defn level-4
  [persister-fn]
  (testing "Compliance level 4, basic support for :delete! mutations"
    (resource/stack-resource-context
     (persist/with-persister (persister-fn)
       (let [[item0] (persist/assoc! {:resource/type :one} :a 1 :b 2)]
         (is (= {:resource/type :one :a 1 :b 2} (select-keys item0 [:resource/type :a :b])))
         (is (= 1 (count (persist/index :one))))
         (persist/with-persister (persist/latest)
           (is (= 1 (count (persist/index :one)))))
         (persist/delete! (:resource/id item0))
         (is (= 0 (count (persist/index :one))))
         (persist/with-persister (persist/latest)
           (is (= 0 (count (persist/index :one)))))
         (is (thrown? Throwable (persist/dissoc! item0 :a :b))))))))

(defn level-5
  [persister-fn]
  (testing "Compliance level 5, basic support for :update! mutations"
    (resource/stack-resource-context
     (persist/with-persister (persister-fn)
       (let [[item] (persist/assoc! {:resource/type :one} :a 1 :b 2)
             [updated-item] (persist/update! item
                                             [:a :add 1]
                                             [:b :sub 5]
                                             [:c :or 1]
                                             [:c :add 1]
                                             [:c :or 1])]
         (is (= {:resource/type :one :a 2 :b -3 :c 2}
                (select-keys updated-item [:resource/type :a :b :c])))
         (is (thrown? Throwable (persist/update! item [:a :< 2])))
         (let [new-item (first (persist/update! item [:a :set -5]))]
           (is (= {:resource/type :one :a -5 :b -3 :c 2}
                  (select-keys new-item [:resource/type :a :b :c])))))))))

(defn level-6
  [persister-fn]
  (testing "Compliance level 6, touchy :assoc!"
    (resource/stack-resource-context
     (persist/with-persister (persister-fn)
       (let [item0 (-> (first (persist/assoc! {:resource/type :one} :a 1 :b 2))
                       (assoc :b 3))
             item1 (first (persist/assoc! item0 :c 4))]
         (is (= {:resource/type :one :a 1 :b 2 :c 4}
                (select-keys item1 [:resource/type :a :b :c])))
         (let [item2 (first (persist/assoc! (assoc item1 :b 5)))]
           (is (= {:resource/type :one :a 1 :b 5 :c 4}
                  (select-keys item2 [:resource/type :a :b :c])))))))))

(defn level-7
  [persister-fn]
  (testing "Compliance level 7, read after write in nested transaction"
    (resource/stack-resource-context
     (persist/with-persister (persister-fn)
       (let [resource-id (UUID/randomUUID)
             resource {:resource/type :one
                       :resource/id resource-id
                       :a 1}
             read-write-fn #(get (persist/index (:resource/type resource)) resource-id)]
         (try
           (persist/with-transaction
             (persist/with-transaction
               (persist/assoc! resource))
             (is (= resource (read-write-fn)))
             (throw nil))
           (catch Throwable t))
         (is (nil? (read-write-fn))))))))

(defn level-8
  [persister-fn]
  (testing "Compliance level 8, concurrent creation semantics")
  (let [persister (persister-fn)
        num-resources 100
        uuid-seq (repeatedly num-resources #(UUID/randomUUID))]
    (resource/stack-resource-context
     (->> uuid-seq
          (map-indexed vector)
          ;; Simulate behavior of a ton of requests
          (pmap
           (fn [[idx uuid]]
             (persist/with-persister
               (persist/latest persister)
               (let [new-resource {:resource/type :test-resource
                                   :resource/id uuid
                                   :data/p1 idx
                                   :data/p2 (* 2 idx)}
                     created (->> (persist/with-transaction
                                    (persist/assoc! new-resource))
                                  (map (juxt :resource/id identity))
                                  (into {}))]
                 (is (not (nil? (get created uuid))))
                 (is (not (nil? (get (persist/index)
                                     uuid))))))))
          doall)
     (is (= num-resources (count (persist/with-persister
                                   (persist/latest persister)
                                   (persist/index))))))))

(defn level-9
  [persister-fn]
  (testing "Compliance level 9, the filter function can re-write resources on read."
    (resource/stack-resource-context
     (persist/with-persister (persister-fn)
       (persist/assoc! {:resource/type :one} :property "1")
       (let [rewrite-fn (fn [resource]
                          (update resource :property #(Integer. %)))
             resource (first (persist/query :one rewrite-fn [:realize [:select :*]]))]
         (is (= 1 (:property resource))))))))
