(ns tech.verify.persist
  (:require [tech.resource :as resource]
            [tech.persist :as persist]
            [tech.persist.protocols :as persist-proto]
            [clojure.test :refer :all])
  (:import [java.util UUID]))


(defn base-persist
  [persister-fn]
  (resource/stack-resource-context
   (persist/with-persister (persister-fn)
     (let [item0 (first (persist/assoc! {:resource/type :one} :a 1 :b 2))
           res-id (:resource/id item0)
           item1 (first (persist/assoc! item0 :a 3))]
       (is (= {:resource/type :one :a 1 :b 2} (select-keys item0 [:resource/type :a :b])))
       (is (= (:resource/id item0) (:resource/id item1)))
       (let [[next-item item2] (persist/with-transaction
                                 (persist/assoc! {:resource/type :one} :a 1 :b 2)
                                 (persist/assoc! item0 :a 4)
                                 ;;check live reads from the temp index
                                 (is (= 4 (get-in (persist/index) [(:resource/id item0) :a]))))
             ;;Order is not guaranteed
             [next-item item2] (if (= (:resource/id item0) (:resource/id item2))
                                 [next-item item2]
                                 [item2 next-item])]
         (is (= {:resource/type :one :a 1 :b 2} (select-keys next-item [:resource/type :a :b])))
         (is (= (:resource/id item0) (:resource/id item2)))
         (is (= {:resource/type :one :a 4 :b 2} (select-keys item2 [:resource/type :a :b]))))
       (let [[item3] (persist/dissoc! item1 :b)]
         (is (= {:resource/type :one :a 4} (select-keys item3 [:resource/type :a :b]))))
       (let [_ (persist/delete! (:resource/id item0))]
         (is (= 1 (count (persist/index :one))))
         (is (thrown? Throwable (persist/dissoc! item0 :a :b)))
         ;;Just make sure this works at all
         (persist/with-persister (persist/latest)
           (is (= 1 (count (persist/index :one))))))
       (let [[rid item] (first (persist/index :one))
             new-item (first (persist/update! item
                                              [:a :add 1]
                                              [:b :sub 5]
                                              [:c :or 1]
                                              [:c :add 1]
                                              [:c :or 1]))]
         (is (= {:resource/type :one :a 2 :b -3 :c 2}
                (select-keys new-item [:resource/type :a :b :c])))
         (is (thrown? Throwable (persist/update! item [:a :< 2])))
         (let [new-item (first (persist/update! item [:a :set -5]))]
           (is (= {:resource/type :one :a -5 :b -3 :c 2}
                  (select-keys new-item [:resource/type :a :b :c])))))))))


(defn assoc-detail
  [persister-fn]
  (resource/stack-resource-context
   (persist/with-persister (persister-fn)
     (testing "The item properties are only used upon create."
       (let [item0 (-> (first (persist/assoc! {:resource/type :one} :a 1 :b 2))
                       (assoc :b 3))
             item1 (first (persist/assoc! item0 :c 4))]
         (is (= {:resource/type :one :a 1 :b 2 :c 4}
                (select-keys item1 [:resource/type :a :b :c])))
         (let [item2 (first (persist/assoc! (assoc item1 :b 5)))]
           (is (= {:resource/type :one :a 1 :b 5 :c 4}
                  (select-keys item2 [:resource/type :a :b :c])))))))))


(defn nested-transaction-read-writes
  [persister-fn]
  (resource/stack-resource-context
    (persist/with-persister (persister-fn)
      (testing "We are able to read our own writes after a nested transaction."
        (let [resource-id (UUID/randomUUID)
              resource {:resource/type :one
                        :resource/id resource-id
                        :a 1}
              read-write-fn #(get (persist/index (:resource/type resource)) resource-id)]
          (try
            (persist/with-transaction
              (persist/with-transaction
                (persist/assoc! resource))
              (is (= resource (read-write-fn)))
              (throw nil))
            (catch Throwable t))
          (is (nil? (read-write-fn))))))))


(defn concurrent-creation
  [persister-fn]
  (let [persister (persister-fn)
        num-resources 100
        uuid-seq (repeatedly num-resources #(UUID/randomUUID))]
    (resource/stack-resource-context
     (->> uuid-seq
          (map-indexed vector)
          ;;Mirror behavior of a ton of requests
          (pmap
           (fn [[idx uuid]]
             (persist/with-persister
               (persist-proto/latest persister)
               (let [new-resource {:resource/type :test-resource
                                   :resource/id uuid
                                   :data/p1 idx
                                   :data/p2 (* 2 idx)}
                     created (->> (persist/with-transaction
                                    (persist/assoc! new-resource))
                                  (map (juxt :resource/id identity))
                                  (into {}))]
                 (is (not (nil? (get created uuid))))
                 (is (not (nil? (get (persist/index)
                                     uuid))))))))
          doall)
     (is (= num-resources (count (persist/with-persister
                                   (persist-proto/latest persister)
                                   (persist/index))))))))

(defn parse-on-read
  [persister-fn]
  (resource/stack-resource-context
   (persist/with-persister (persister-fn)
     (testing "The filter function can re-write resources on read."
       (persist/assoc! {:resource/type :one} :property "1")
       (let [rewrite-fn (fn [resource]
                          (update resource :property #(Integer. %)))
             resource (first (persist/query :one rewrite-fn [:realize [:select :*]]))]
         (is (= 1 (:property resource))))))))
