(ns tech.persist.utils
  (:require [tech.persist.atom :as persist-atom])
  (:refer-clojure :exclude [assoc! dissoc!]))


(defn persist-trans-data
  []
  (atom nil))


(defn inc-trans!
  [trans-data* main-index]
  (swap! trans-data* update :trans-stack
         (fn [trans-stack]
           (conj trans-stack
                 {:mutations []
                  :index (or (:index (peek trans-stack))
                             main-index
                             {})}))))


(defn dec-trans!
  [trans-data* mutation-fn!]
  (let [[old-data new-data]
        (swap-vals! trans-data*
                    (fn [{:keys [trans-stack]}]
                      (let [{:keys [index mutations]} (peek trans-stack)
                            remaining (seq (pop trans-stack))]
                        (when remaining
                          (let [{remaining-mutations :mutations
                                 remaining-index :index} (peek remaining)]
                            {:trans-stack (conj (pop remaining)
                                                {:mutations (vec (concat remaining-mutations
                                                                         mutations))
                                                 :index index})})))))]
    (when (= 1 (count (:trans-stack old-data)))
      (assert (not (seq (:trans-stack new-data))))
      (-> (:trans-stack old-data)
          first
          :mutations
          mutation-fn!))))


(defn rollback-trans!
  [trans-data*]
  (swap! trans-data* update :trans-stack pop))


(defn add-mutation!
  [trans-data* mutation]
  (swap! trans-data* update :trans-stack
         (fn [trans-stack]
           (when-not (peek trans-stack)
             (throw (ex-info "Transaction is not open" {})))
           (let [{:keys [mutations index]} (peek trans-stack)]
             (conj (pop trans-stack)
                   {:mutations (conj mutations mutation)
                    :index (-> (persist-atom/apply-mutation [index #{}] mutation)
                               first)})))))


(defn temp-index
  [trans-data*]
  (when trans-data*
    (-> @trans-data*
        :trans-stack
        peek
        :index)))


(defn assoc!
  [trans-data* item key-val-seq]
  (add-mutation! trans-data* {:mutation-type :assoc!
                              :item item
                              :key-val-seq key-val-seq}))


(defn update!
  [trans-data* item command-seq]
  (add-mutation! trans-data* {:mutation-type :update!
                              :item item
                              :command-seq command-seq}))


(defn dissoc!
  [trans-data* item key-seq]
  (add-mutation! trans-data* {:mutation-type :dissoc!
                              :item item
                              :key-seq key-seq}))


(defn delete!
  [trans-data* item]
  (add-mutation! trans-data* {:mutation-type :delete!
                              :item item}))
