(ns tech.persist.atom
  (:import [java.util UUID]))


(defmulti apply-mutation
  (fn [world mutation]
    (:mutation-type mutation)))


(defmethod apply-mutation :assoc!
  [[world change-set] {:keys [item key-val-seq]}]
  (let [key-val-map (if-not (map? key-val-seq)
                      (into {} key-val-seq)
                      key-val-seq)
        item (merge item key-val-map)
        res-id (or (:resource/id item)
                   (UUID/randomUUID))
        old-item (-> (or (get world res-id)
                         item)
                      (assoc :resource/id res-id))
        world (assoc world res-id (merge old-item key-val-map))]
    [world (conj change-set res-id)]))


(defn- ensure-exists
  [item item-key]
  (if-let [retval (get item item-key)]
    retval
    (throw (ex-info (format "Failed to get value %s" item-key)
                    {:item item
                     :item-key item-key}))))



(defmulti apply-update
  (fn [item item-key cmd-name cmd-args]
    cmd-name))

(defn- do-op
  [item item-key item-fn cmd-args]
  (apply item-fn (ensure-exists item item-key) cmd-args))


(defmethod apply-update :add
  [item item-key cmd-name cmd-args]
  (do-op item item-key + cmd-args))


(defmethod apply-update :sub
  [item item-key cmd-name cmd-args]
  (do-op item item-key - cmd-args))


(defmethod apply-update :mul
  [item item-key cmd-name cmd-args]
  (do-op item item-key * cmd-args))


(defmethod apply-update :div
  [item item-key cmd-name cmd-args]
  (do-op item item-key / cmd-args))


(defmethod apply-update :or
  [item item-key cmd-name cmd-args]
  (get item item-key (first cmd-args)))


(defn- check-consistency
  [item item-key check-fn cmd-name cmd-args]
  (let [retval (ensure-exists item item-key)]
    (when-not (check-fn retval (first cmd-args))
      (throw (ex-info "Consistency check failed"
                      {:item item
                       :item-key item-key
                       :cmd-name cmd-name
                       :cmd-value (first cmd-args)})))
    retval))


(defmethod apply-update :>=
  [item item-key cmd-name cmd-args]
  (check-consistency item item-key >= cmd-name cmd-args))


(defmethod apply-update :>
  [item item-key cmd-name cmd-args]
  (check-consistency item item-key > cmd-name cmd-args))


(defmethod apply-update :=
  [item item-key cmd-name cmd-args]
  (check-consistency item item-key = cmd-name cmd-args))


(defmethod apply-update :not=
  [item item-key cmd-name cmd-args]
  (check-consistency item item-key not= cmd-name cmd-args))


(defmethod apply-update :<
  [item item-key cmd-name cmd-args]
  (check-consistency item item-key < cmd-name cmd-args))


(defmethod apply-update :<=
  [item item-key cmd-name cmd-args]
  (check-consistency item item-key <= cmd-name cmd-args))


(defmethod apply-update :set
  [item item-key cmd-name cmd-args]
  (when-not (first cmd-args)
    (throw (ex-info "No value to set attribute to" {})))
  (first cmd-args))


(defn- get-item-for-update
  [world item]
  (let [res-id (cond
                 (uuid? item)
                 item
                 (map? item)
                 (:resource/id item)
                 :else
                 (item :resource/id))
        _ (when-not res-id
            (throw (ex-info "Failed to get resource id from item"
                            {:item item})))
        update-item (get world res-id)
        _ (when-not update-item
            (throw (ex-info "No world mapped to resource id"
                            {:resource/id res-id})))]
    update-item))



(defmethod apply-mutation :update!
  [[world change-set] {:keys [item command-seq]}]
  (let [{res-id :resource/id
         :as update-item}
        (get-item-for-update world item)
        new-item (reduce (fn [new-item cmd]
                           (let [{:keys [item-key cmd-name cmd-args]} cmd]
                             (assoc new-item item-key
                                    (apply-update new-item item-key cmd-name cmd-args))))
                         update-item
                         command-seq)]
    [(assoc world res-id new-item)
     (conj change-set res-id)]))


(defmethod apply-mutation :dissoc!
  [[world change-set] {:keys [item key-seq]}]
  (let [{res-id :resource/id
         :as update-item}
        (get-item-for-update world item)]
    [(assoc world res-id (apply dissoc update-item key-seq))
     (conj change-set res-id)]))


(defmethod apply-mutation :delete!
  [[world change-set] {:keys [item]}]
  (let [res-id (cond
                 (uuid? item)
                 item
                 (map? item)
                 (:resource/id item)
                 :else
                 (item :resource/id))
        _ (when-not res-id
            (throw (ex-info "Failed to get resource id from item"
                            {:item item})))]
    [(dissoc world res-id) (conj change-set res-id)]))


(defn perform-mutations!
  [index* mutation-seq]
  (loop [old-world @index*]
    (let [[world change-set] (reduce apply-mutation
                                     [old-world []]
                                     mutation-seq)]
      (if (compare-and-set! index* old-world world)
        (map #(get world %) (distinct change-set))
        (recur @index*)))))
