(ns tech.persist
  (:require [tech.resource :as resource]
            [tech.persist.protocols :as persist-proto]
            [tech.persist.utils :as utils]
            [tech.query :as query]
            [tech.config.core :as config]
            [tech.parallel.require :as parallel-require])
  (:refer-clojure :exclude [assoc! dissoc!]))


(def ^:dynamic *persister-fn* nil)


(defmacro with-persister-fn
  [persister-fn & body]
  `(with-bindings {#'*persister-fn* ~persister-fn}
     ~@body))


(defmacro with-persister
  [persister & body]
  `(with-persister-fn (constantly ~persister) ~@body))


(defn persister
  []
  (*persister-fn*))


(def ^:dynamic *open-transaction* nil)


(defmacro with-transaction
  "Open (or nest further) a transaction.  Transactions are nesting and the mutation
will only occur when last one is closed.  Use this macro to do this safely."
  [& body]
  `(with-bindings {#'*open-transaction* (or *open-transaction*
                                            (utils/persist-trans-data))}
     (let [engine# (persister)]
       (utils/inc-trans! *open-transaction* (persist-proto/index engine#))
       (try
         ~@body
         (utils/dec-trans! *open-transaction* (persist-proto/get-mutation-fn! engine#))
         (catch Throwable e#
           (utils/rollback-trans! *open-transaction*)
           (throw e#))))))




(defn assoc!
  "Associate new or updated properties to an object.

  If the object does not exist, then all the properties of the item merged with the
  updates are set.  In this case both resource/id and resource/type must be set.

  If the item does exist, then only updates are set; the latest item is read from the
  datasource using the resource id of the item.

  If the item exists, then item may just be a resource id.  This will cause an error in
  the case where the item does not exist.

  Unlike assoc, the first arg can be a full map.  If there are more arguments then they
  are treated like identically to assoc.
  "
  ([item]
   (assoc! item item))
  ([item arg & args]
   (with-transaction
     (utils/assoc! *open-transaction* item (if-not (seq args)
                                             arg
                                             (->> (concat [arg] args)
                                                  (partition 2)
                                                  (map vec)))))))


(def supported-commands
  #{:add :sub :mul :div :or :> :< := :!=})


(defn update!
  "Update an item by performing operations on it.
Command format is:
[item-key command-name & cmd-args].

The 'or' form of the arithmetic commands or whatever is in the object
with the first argument of the cmd-args."
  [item & command-seq]
  (let [cmd-map-seq
        (->> command-seq
             (map (fn [cmd-data]
                    (when (< (count cmd-data) 2)
                      (throw (ex-info "Update commands must have at least length 2"
                                      {:cmd-data cmd-data})))
                    (let [item-key (first cmd-data)
                          cmd-name (second cmd-data)
                          cmd-args (seq (drop 2 cmd-data))]
                      {:item-key item-key
                       :cmd-name cmd-name
                       :cmd-args cmd-args})))
             doall)]
    (with-transaction
      (utils/update! *open-transaction* item cmd-map-seq))))



(defn dissoc!
  "Dissoc these keys from this item."
  [item & key-seq]
  (with-transaction
    (utils/dissoc! *open-transaction* item key-seq)))


(defn delete!
  [item]
  (with-transaction
    (utils/delete! *open-transaction* item)))


(defn- raw-index
  [resource-type]
  (cond->> (or (utils/temp-index *open-transaction*)
               (persist-proto/index (persister)))
    resource-type
    (filter #(= resource-type (get (second %) :resource/type)))))


(defn index
  [& [resource-type]]
  (let [retval (raw-index resource-type)]
    (if-not (map? retval)
      (into {} retval)
      retval)))


(defn query
  [resource-type filter-fn query-args]
  (let [index (cond->> (raw-index resource-type)
                filter-fn
                (map (fn [[id resource]]
                       (if-let [filter-result (filter-fn resource)]
                         (if (map? filter-result)
                           [id filter-result]
                           [id resource]))))
                :always (remove nil?))
        index (if-not (map? index)
                (into {} index)
                index)]
    (query/query {:primary-index index} query-args)))


(defn latest
  []
  (persist-proto/latest (persister)))


(defonce ^:dynamic *ring-persister* (atom nil))


(defn ring-handler
  "A ring handler that updates the persister to the latest and ensures the
  global ring-persister atom points to the latest persister."
  [next-handler]
  (let [current-persister (persister)]
    (when-not current-persister
      (throw (ex-info "Persist system unbound"
                      {})))
    (let [handler-atom (atom current-persister)]
      (fn [req]
        ;;Get the latest from the datomic or duratom system
        ;;This also means that if by some crazy flaw the persister's transaction gets
        ;;corrupted the corruption will be localized to one request
        (with-persister (swap! handler-atom persist-proto/latest)
          (reset! *ring-persister* (persister))
          (next-handler req))))))


(defn ring-persister
  []
  @*ring-persister*)


(defn set-ring-persister-root!
  "Called in your main namespace always.  Sets up repl access to the ring persister"
  []
  (reset! *ring-persister* (persister))
  (alter-var-root  #'*persister-fn* (constantly ring-persister)))


(defn config-persister
  "Create a persister from environment config variables"
  []
  (let [datomic-uri (config/unchecked-get-config :datomic-uri)]
    (if (> (count datomic-uri) 1)
      ((parallel-require/require-resolve 'tech.persist.datomic/persister)
       datomic-uri)
      ((parallel-require/require-resolve 'tech.persist.duratom/persister)
       (config/get-config :index-nippy-path)))))


(defn set-config-persister!
  []
  (with-persister
    (config-persister)
    (set-ring-persister-root!)))
