(ns tech.thomascothran.pavlov.event.publisher.linear
  "The linear publisher is asynchronous but ensures the
  order of events send to the handler"
  (:require [tech.thomascothran.pavlov.bprogram.proto :as bprogram]
            [tech.thomascothran.pavlov.event :as event]
            [tech.thomascothran.pavlov.event.publisher.proto :as publisher])
  (:import [java.util.concurrent LinkedBlockingQueue]))

(extend-protocol bprogram/BProgramQueue
  LinkedBlockingQueue
  (conj [this event]
    (.put this event))
  (pop [this] (.take this)))

(defn- notify-listeners!
  [listeners event]
  (doseq [[k listener] listeners]
    (try (listener event)
         (catch Throwable _
           (swap! listeners dissoc k)))))

(defn- run-notify-loop!
  [publisher opts]
  (let [killed?            (get opts :killed?)
        subscribers        (get publisher :subscribers)
        on-stopped         (get opts :on-stopped)
        notification-queue publisher]
    (loop [event (bprogram/pop notification-queue)]
      (when-not (killed?)
        (notify-listeners! @subscribers event)
        (if (event/terminal? event)
          (on-stopped)
          (recur (bprogram/pop notification-queue)))))))

(defn- notify!
  [publisher event]
  (conj (get publisher :queue) event))

(defn- subscribe!
  [publisher k f]
  (swap! (get publisher :subscribers) assoc k f))

(defn make-publisher!
  "Make an asynchronous, linear publisher
  
  opts:
  -----
  - `:subscribers`: a map of subscriber key -> subscriber function
  - `:killed`: a deferred delivered only when the bprogram has been killed.
  - `:on-stopped`: a 0 arity function to call when the program has stopped"
  [opts]
  (with-meta {:queue       (get opts :queue (LinkedBlockingQueue.))
              :subscribers (get opts :subscribers (atom {}))
              :stopped     (promise)}
    {`publisher/start!
     (fn [this] (run-notify-loop! this opts))
     `publisher/notify! notify!
     `publisher/subscribe! subscribe!
     `publisher/subscribers! #(-> (get % :subscribers) deref)}))
