(ns corvus.sentry.util
  (:require [clojure.java.io :as io]
            [clojure.stacktrace :as stk]
            [clojure.string :as str]
            [corvus.util :as util]
            [prone.stacks :as stkx]))

(defn parse
  [{:keys [data-source-name]}]
  (let [[scheme+credentials host+port+path] (some-> data-source-name
                                                    (str/split #"@"))
        [protocol credentials] (some-> scheme+credentials
                                       (str/split #"://"))
        [username password] (some-> credentials
                                    (str/split #":"))]
    (let [public-key  (some-> username
                              not-empty)
          secret-key  (some-> password
                              not-empty)
          sentry-host (some-> host+port+path
                              (str/split #"/")
                              butlast)
          project-id  (some-> host+port+path
                              (str/split #"/")
                              last
                              util/string->integer)]
      (when (and public-key
                 sentry-host
                 project-id)
        (let [sentry {:data-source {:public-key public-key
                                    :store-api  (->> sentry-host
                                                     (conj ["store" project-id "api"])
                                                     reverse
                                                     flatten
                                                     (str/join "/")
                                                     (format "%s://%s" protocol))}}]
          (if secret-key
            (assoc-in sentry [:data-source :secret-key] secret-key)
            sentry))))))

(defn- in-application?
  [application-name package]
  (->> application-name
       (some #(.startsWith package %))
       boolean))

(defn- class->source
  [class-path-url line-number]
  (some-> class-path-url
          (io/resource)
          slurp
          (str/split #"\n")
          (#(drop (- line-number 6) %))
          (#(take 11 %))))

(defn- frame->source
  [application-namespaces
   {:keys [class-path-url
           line-number
           file-name
           package
           method-name]}]
  (let [source (class->source class-path-url line-number)]
    {:in_app       (in-application? application-namespaces package)
     :filename     file-name
     :lineno       line-number
     :function     (str package "/" method-name)
     :context_line (nth source 5 nil)
     :pre_context  (take 5 source)
     :post_context (drop 6 source)}))

(defn stacktrace
  [event
   ^Exception error
   & [application-namespaces]]
  (let [{:keys [frames]
         :as   stacks} (-> error
                           stk/root-cause
                           stkx/normalize-exception)
        frames (->> frames
                    reverse
                    (map #(frame->source application-namespaces %)))]
    (assoc event
           :exception [{:type       (get stacks :type)
                        :value      (get stacks :message)
                        :stacktrace {:frames frames}}])))
