(ns corvus.sentry.http
  (:require [clj-http.client :as http]
            [clojure.string :as str]
            [cheshire.core :as json])
  (:import [java.net InetAddress]
           [java.sql Timestamp]
           [java.util Date]
           [java.util UUID]))

(defn- make-sentry-auth-header
  [timestamp
   {:keys [public-key
           secret-key]}]
  (->> ["Sentry sentry_version=2.0"
        (format "sentry_client=tech.gojek/corvus")
        (format "sentry_timestamp=%s" timestamp)
        (format "sentry_key=%s" public-key)
        (when secret-key
          (format "sentry_secret=%s" secret-key))]
       (remove nil?)
       (str/join ", ")))

(defn capture!
  [{:keys [data-source]} event]
  (let [{:keys [store-api]} data-source
        event              (-> {:level       "error"
                                :platform    "Clojure"
                                :server_name (-> (InetAddress/getLocalHost)
                                                 .getHostName)
                                :timestamp   (-> (Date.)
                                                 .getTime
                                                 (Timestamp.)
                                                 .toString)}
                               (merge event)
                               (merge {:event_id (-> (UUID/randomUUID)
                                                     (str/replace #"-" ""))}))
        sentry-auth-header (-> event
                               (get :timestamp)
                               (make-sentry-auth-header data-source))
        event-json         (-> event
                               (dissoc :timestamp)
                               (json/generate-string))]
    (http/post store-api
               {:throw-exceptions false
                :headers          {"User-Agent"    "tech.gojek/corvus"
                                   "X-Sentry-Auth" sentry-auth-header}
                :body             event-json})))
