# Corvus

<p align="center">
  <a href="https://travis-ci.org/gojektech/corvus">
    <img src="https://travis-ci.org/gojektech/corvus.svg?branch=master" alt="Build Status" />
  </a>
  <a href='https://coveralls.io/github/gojektech/corvus?branch=master'>
    <img src='https://coveralls.io/repos/github/gojektech/corvus/badge.svg?branch=master' alt='Coverage Status' />
  </a>
</p>

* [Description](#description)
* [Installation](#installation)
* [Usage](#usage)
* [Contributing](#contributing)
* [Development](#development)
* [License](#license)

## Description

[Sentry][sentry] client in [Clojure][clojure] to report error events asynchronously.

## Installation

Add to `project.clj`'s dependencies,
```
[tech.gojek/corvus "1.0.0"]
```

## Usage

0. Firstly, require `corvus.core` in application's namespace.

```clojure
(ns example.app
  (:require [corvus.core :as corvus])
  (:import [java.util.concurrent TimeUnit]))
```

1. And now, start with setting up Sentry captor with synchronous executor.

```clojure
(def captor
  (corvus/make-captor
    {:data-source-name "https://__public-key_:__secret-key__@example.com/42" ; (required) a valid DSN, need to push/store events
     :logger-fn        (fn [_level _message & _more])}      ; (optional) default: clojure.tools.logging/logp
    {:namespace   "example.app"                             ; (required) Application's namespace, need to generate stacktraces
     :environment "test"}))                                 ; (required) Application's runtime environment, need to classify events
```

Or, captor with asynchronous executor.

```clojure
(def captor
  (corvus/make-captor
    {:data-source-name "https://__public-key_:__secret-key__@example.com/42" ; (required) a valid DSN, need to push/store events
     :logger-fn        (fn [_level _message & _more])}      ; (optional) default: clojure.tools.logging/logp
    {:namespace   "example.app"                             ; (required) Application's namespace, need to generate stacktraces
     :environment "test"}                                   ; (required) Application's runtime environment, need to classify events
    {:async?                                 true           ; (optional) default: false, enables async executor
     :async-executor-workers-idle-count      42             ; (optional) default: 10, async executor's idle workers count
     :async-executor-workers-maximum-count   64             ; (optional) default: 10, async executor's maximum workers count
     :async-executor-workers-keep-alive-time 5              ; (optional) default: 0, keep-alive timeout for workers over idle count
     :async-executor-workers-keep-alive-unit TimeUnit/SECONDS ; (optional) default: TimeUnit/MILLISECONDS, keep-alive timeout's unit
     :async-executor-workers-queue-size      8}))           ; (optional) default: 10, async executor's submitted tasks queue size
```

Also to ensure safe captor's executors shutdown. Add this to `Runtime`'s 
shutdown hook.

```clojure
(-> (Runtime/getRuntime)
    (.addShutdownHook (Thread. #(corvus/kill-captor captor))))
```

Or, if you're using [mount](https://github.com/tolitius/mount).

```clojure
(defstate captor
  :start (corvus/make-captor
            {:data-source-name "https://__public-key_:__secret-key__@example.com/42"
             :logger-fn        (fn [_level _message & _more])}
            {:namespace   "example.app"
             :environment "test"})                           ; add async options if require an async executor  
  :stop (corvus/kill-captor captor))
  
(mount/start captor)

(-> (Runtime/getRuntime)
    (.addShutdownHook (Thread. #(mount/stop))))
``` 

2. And finally, capturing events with Sentry.

```clojure
(corvus/capture! captor 
                 :warn                                       ; (optional nil) error level, default: :error, valid values: :warn|:error
                 error                                       ; (required) java.lang.Throwable type error
                 message-1                                   ; (optional) java.lang.String type message
                 message-2)                                  ; (optional) java.lang.String type message
```

## Contributing

Please! go through our [Contributing Guidelines](./CONTRIBUTING.md) before submitting
a [Pull Request](https://github.com/gojektech/corvus/pulls). 
And also, refer [AUTHORS](./AUTHORS.md) if you wanna reach out to contributors.  

## Development

- Firstly, setup these pre-requisites.
  - [Java SDK](https://www.oracle.com/technetwork/java/javase/downloads/index.html) (>= 8)
  - [Leiningen](https://github.com/technomancy/leiningen) (>= v2.5)
  - [GNU Make](https://www.gnu.org/software/make) (>= v4.2)

- And now, how to build & test this project.
  1. Start with `make clean install` to clean slate and install development 
     dependencies.
  2. Make sure you run `make setup` to install development githooks.
  3. It's always nice to run `make test` triggering unit tests. Optionally, to get 
     overall code coverage run `make coverage`.
  4. Optionally, run `make format` to warn and fix linting errors if there're any ;)  

__NOTE:__ FYI, project's pre-commit hook triggers `make format` and pre-push runs 
`make coverage`. This is to avoid checking-in any linting and testing errors :D

## License

```
Copyright 2018, GO-JEK Tech <http://gojek.tech>

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
```

[sentry]: https://sentry.io
[clojure]: https://clojure.org
