# bulwark

Hystrix for Clojurists 

## Description 
A light macro to wrap code in a Hystrix command, for circuit-breaking and fault tolerance.

## Usage
Add this to your dependencies:
`[tech.gojek/bulwark "1.0.0]`  

`bulwark` exposes a macro called `with-hystrix`, that takes a map of Hystrix configuration and executes the body in a Hystrix command.


```clojure
(let [fallback (fn [_]
                 "failure")]
  ;; Returns "success"
  (bulwark/with-hystrix {;; These three keys identify the Hystrix command 
                         ;; The group key is used to group commands together,
                         ;; for reporting/monitoring. We suggest having a
                         ;; single group key for all commands in your app.
                         :group-key                          "your-app"
                         
                         ;; The command key uniquely identifies the Hystrix command.                        
                         :command-key                        "foo-command"
                         
                         ;; The thread pool key identifies the thread pool
                         ;; to be used to execute your Hystrix command.
                         ;; If two commands have the same thread pool key, 
                         ;; they will share the same thread pool.
                         ;; We suggest using different thread pools for each command.
                         :thread-pool-key                    "foo-thread-pool"
                         
                         :thread-count                       2
                         :breaker-sleep-window-ms            500
                         :breaker-error-threshold-percentage 20
                         :execution-timeout-ms               100
                         :fallback-fn                        fallback}
                        "success"))

(let [fallback
      ;; These are the keys the fallback function is passed.
      (fn [{:keys [response-timed-out?                      
                   failed-execution-exception               ; If an exception was thrown, it is available here
                   circuit-breaker-open?                    
                   semaphore-rejected?                      ; See https://github.com/Netflix/Hystrix/wiki/Configuration#execution.isolation.semaphore.maxConcurrentRequests
                   thread-pool-rejected?                    
                   ]}]
        "failure")]

  ;; Runs the fallback (because of a timeout) and evaluates to its return value, i.e. "failure"
  (bulwark/with-hystrix {:group-key                          "your-app"
                         :command-key                        "foo-command"
                         :thread-pool-key                    "foo-thread-pool"
                         :thread-count                       2
                         :breaker-sleep-window-ms            500
                         :breaker-error-threshold-percentage 20
                         :execution-timeout-ms               100
                         :fallback-fn                        fallback}
                        (Thread/sleep 500)
                        "success"))
```

## License
```
Copyright 2018, GO-JEK Tech <https://gojek.tech>

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
```

