(ns versioniq.main
  (:gen-class)
  (:require
    [unified.response :as r]
    [versioniq.api :as api]
    [versioniq.config :as config]
    [versioniq.log :as log]))


(set! *warn-on-reflection* true)


(def ^{:added "0.0.1"}
  dev?
  "Returns `true` if the current mode is development. Otherwise, `false`."
  (= "true" (System/getenv "versioniq_dev")))


(def ^{:added "0.0.1"}
  codes
  "Mapping of response types with exit status codes."
  {::unexpected-error 124})


(defn exit
  "Terminates the currently running Java Virtual Machine using the specified response type."
  {:added "0.0.1"}
  [type]
  (when-not dev?
    (if-not (r/anomaly? type)
      (System/exit 0)
      (System/exit (get codes type 1)))))


(defn exec
  "Exec a command."
  {:added "0.0.1"}
  [cmd args]
  (if-some [[type res] (->> args
                         api/parse-opts
                         config/read-config
                         (api/invoke cmd (api/filter-args args)))]
    (if (r/anomaly? res)
      (log/error res)
      (log/info res))
    type))


(defn -main
  "Entry point."
  {:added "0.0.1"}
  [& [cmd & args]]
  (let [type
        (try
          (exec cmd args)
          (catch Throwable e
            (if dev?
              (throw e)
              (do
                (log/error "Unexpected error. Please report an issue.")
                (.printStackTrace e)
                ::unexpected-error))))]
    (flush)
    (exit type)))


(comment
  (-main nil) ;; show help
  (-main "help") ;; show help

  (-main "config") ;; show config
  (-main "config" "tag" "prefix") ;; show tag prefix using a `:default` profile
  (-main "config" "tag" "prefix" "--profile" "dev") ;; show tag prefix using a `:dev` profile

  (-main "build") ;; show build info
  (-main "build" "version") ;; show build version
  )
