(ns versioniq.api
  (:require
    [clojure.edn :as edn]
    [clojure.java.io :as io]
    [clojure.string :as str]
    [unified.response :as r]
    [versioniq.helpers :as helpers]))


(set! *warn-on-reflection* true)


(def ^{:added "0.0.1"}
  build
  "Project build info."
  (helpers/try-or
    (->> "versioniq/build.edn"
      io/resource
      slurp
      edn/read-string)
    (constantly nil)))


(def ^{:added "0.0.1"}
  help
  "Project help info."
  (->> [(:tag build)
        ""
        "Usage: versioniq command [parameters]"
        ""
        "Commands:"
        "  init       Init the first version"
        "  list       Show all project versions"
        "  current    Show the current version"
        "  previous   Show the previous version"
        "  next       Show the next version"
        "  up         Bump to the next version"
        "  down       Down to the previous version"
        "  release    Release the current version"
        "  sync       Synchronise all versions"
        "  diff       Show diff between versions"
        ""
        "Internal commands:"
        "  help            Show help"
        "  config [keys]   Show config"
        "  build  [keys]   Show build info"
        ""
        "Options:"
        "  --profile          Read config using the specified profile. By default using `:default` profile"
        "  --root-directory   Read config using the specified root directory. By default using the current directory."
        "  --config-file      Read config using the specified config file name. By default using `.versioniq`"]
    (str/join (System/lineSeparator))))


(def ^{:added "0.0.1"}
  opts
  "Project opts."
  {"--profile"        :profile
   "--root-directory" :root-directory
   "--config-file"    :config-file})


(def ^{:added "0.0.1"}
  double-dash?
  "Returns `true` if the given argument is a double dash (--) separator. Otherwise, `false`."
  (partial = "--"))


(def ^{:added "0.0.1"}
  opt?
  "Returns `true` if the given `x` is opt. Otherwise, `false`."
  (let [opts (->> opts keys set)]
    (fn [x]
      (contains? opts x))))


(defn parse-opts
  "Returns a parsed opts."
  {:added "0.0.1"}
  [args]
  (loop [acc {}
         [x & xs :as args] args]
    (if-not (seq args)
      acc
      (cond
        (double-dash? x) (recur acc xs)
        (not (opt? x)) (recur acc xs)
        :else (recur (assoc acc (get opts x) (first xs)) (next xs))))))


(defn filter-args
  "Returns args without opts."
  {:added "0.0.1"}
  [args]
  (loop [acc []
         [x & xs :as args] args]
    (if-not (seq args)
      acc
      (cond
        (double-dash? x) (recur acc xs)
        (not (opt? x)) (recur (conj acc x) xs)
        :else (recur acc (next xs))))))


(defn internal-cmd?
  "Returns `true` if the given `x` is an internal command. Otherwise, `false`."
  {:added "0.0.1"}
  [x]
  (contains? #{:help :config :build} x))


(defmulti invoke
  "Invokes a command."
  {:arglists '([cmd args config])
   :added    "0.0.1"}
  (fn [cmd _ {:keys [storage]
              :or   {storage :git}}]
    (let [cmd (helpers/keywordize cmd)]
      (cond
        (nil? cmd) :help
        (internal-cmd? cmd) cmd
        :else [storage cmd]))))


(defmethod invoke :default [cmd _ _]
  (r/as-error (format "Unknown command: %s" (name cmd))))


(defmethod invoke :help [_ _ _]
  (r/as-success help))


(defmethod invoke :config [_ args config]
  (if-some [v (->> args
                (mapv helpers/keywordize)
                (get-in config))]
    (r/as-success (helpers/pretty-print v))
    (r/as-not-found (str/join \space (cons "Unknown parameters of the `config` command:" args)))))


(defmethod invoke :build [_ args _]
  (if-some [v (->> args
                (mapv helpers/keywordize)
                (get-in build))]
    (r/as-success (helpers/pretty-print v))
    (r/as-not-found (str/join \space (cons "Unknown parameters of the `build` command:" args)))))
