(ns unified.response
  (:refer-clojure :exclude [type error?])
  (:import
    (clojure.lang
      PersistentList
      PersistentVector)))


;;
;; Protocols
;;

(defprotocol IResponse
  "A unified response protocol."
  :extend-via-metadata true
  (error? [this] "Returns `true` if `type` of a unified response is an anomaly. Otherwise, `false`.")
  (type [this] "Returns `type` of a unified response."))


(defprotocol IResponseBuilder
  "A unified response builder protocol."
  :extend-via-metadata true
  (as-response [this type] "Returns a unified response with the given response type."))



;;
;; Anomalies
;;

(defonce ^{:doc "Registry of anomalies.", :added "0.0.1"}
  *anomalies
  (atom
    #{::error
      ::warning
      ::exception
      ::unavailable
      ::interrupted
      ::incorrect
      ::unauthorized
      ::forbidden
      ::not-found
      ::unsupported
      ::conflict
      ::busy
      ::unknown}))


(defn anomaly?
  "Returns `true` if `x` is an anomaly or derived from `::error`. Otherwise, `false`."
  {:added "0.0.1"}
  [x]
  (or
    (boolean (@*anomalies x))
    (isa? x ::error)))



;;
;; Error response builders
;;

(defn as-error
  "Returns a unified `error` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::error))


(defn as-warning
  "Returns a unified `warning` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::warning))


(defn as-exception
  "Returns a unified `exception` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::exception))


(defn as-unavailable
  "Returns a unified `unavailable` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::unavailable))


(defn as-interrupted
  "Returns a unified `interrupted` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::interrupted))


(defn as-incorrect
  "Returns a unified `incorrect` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::incorrect))


(defn as-unauthorized
  "Returns a unified `unauthorized` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::unauthorized))


(defn as-forbidden
  "Returns a unified `forbidden` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::forbidden))


(defn as-not-found
  "Returns a unified `not-found` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::not-found))


(defn as-unsupported
  "Returns a unified `unsupported` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::unsupported))


(defn as-conflict
  "Returns a unified `conflict` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::conflict))


(defn as-busy
  "Returns a unified `busy` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::busy))


(defn as-unknown
  "Returns a unified `unknown` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::unknown))



;;
;; Success response builders
;;

(defn as-success
  "Returns a unified `success` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::success))


(defn as-found
  "Returns a unified `found` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::found))


(defn as-created
  "Returns a unified `created` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::created))


(defn as-updated
  "Returns a unified `updated` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::updated))


(defn as-deleted
  "Returns a unified `deleted` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::deleted))


(defn as-accepted
  "Returns a unified `accepted` response."
  {:added "0.0.1"}
  [x]
  (as-response x ::accepted))



;;
;; Extend protocols
;;

(extend-protocol IResponse
  nil
  (error? [_] false)
  (type [_] nil)

  Object
  (error? [_] false)
  (type [_] nil)

  PersistentList
  (error? [this]
    (anomaly? (type this)))
  (type [this]
    (first this))

  PersistentVector
  (error? [this]
    (anomaly? (type this)))
  (type [this]
    (first this)))


(extend-protocol IResponseBuilder
  nil
  (as-response [this type]
    [type this])

  Object
  (as-response [this type]
    [type this])

  PersistentList
  (as-response [this type]
    (if (error? this)
      (assoc this 0 type)
      [type this]))

  PersistentVector
  (as-response [this type]
    (if (error? this)
      (assoc this 0 type)
      [type this])))
