(ns ninja.cli.core
  (:require
    [ninja.cli.protocols :as cli.proto]
    [ninja.cli.protocols.arguments :as argument.proto]
    [ninja.cli.protocols.commands :as command.proto]
    [ninja.cli.protocols.params :as param.proto]
    [ninja.cli.specs :refer [parse-cli valid? explain humanize]]))


(defn ->param
  [{:param/keys [name options]}]
  ^{:type ::param.proto/param}
  (reify param.proto/Param
    (name [_] name)
    (options [_] options)))


(defn ->argument
  [{:argument/keys [name options params]}]
  ^{:type ::argument.proto/argument}
  (reify argument.proto/Argument
    (name [_] name)
    (options [_] options)
    (params [_] (mapv ->param params))))


(defn ->command
  [{:command/keys [name options arguments]}]
  ^{:type ::command.proto/command}
  (reify command.proto/Command
    (name [_] name)
    (options [_] options)
    (arguments [_] (mapv ->argument arguments))))


(defn ->cli
  [{:cli/keys [name options commands] :as cli}]
  ^{:type ::cli.proto/cli}
  (reify cli.proto/CLI
    (name [_] name)
    (options [_] options)
    (commands [_] (mapv ->command commands))
    (describe [_] cli)))


(defn build-cli
  [cli]
  (let [parsed (parse-cli cli)]
    (if (= :cli/invalid parsed)
      (throw (ex-info "invalid cli" {:errors (explain cli)
                                     :pretty-errors (humanize cli)}))
      (->cli parsed))))


(def ninja-cli
  [:ninja/cli {:cli :options}
   [:init
    [:workspace]
    [:project]
    [:module]
    [:library]]
   [:add
    [:project]
    [:module]
    [:library]]
   [:install]
   [:bootstrap]
   [:publish]
   [:info]
   [:version]])


(comment

  (valid? ninja-cli)
  (explain ninja-cli)
  (parse-cli ninja-cli)
  (def c (build-cli ninja-cli))
  (cli.proto/name c)
  (cli.proto/options c)
  (cli.proto/commands c)
  (cli.proto/describe c)
  (mapv command.proto/name (cli.proto/commands c))

  (humanize {:locale :ru} ninja-cli)
  )



;; nj init                          -> readline type -> readline name
;; nj init --type workspace         -> readline name
;;
;; nj init workspace                -> readline name
;; nj init workspace team.sultanov

;; nj init project                  -> readline name
;; nj init project sleepy-kitty
;; nj init module                   -> readline name
;; nj init module server

