(ns tandem.settings

  "Parse the config.json configuration file and manage settings across the
  entire application.  Also understands the current state of the server, whether
  it is in development, test, or production mode.

  Settings are hierarchical.  Inside the settings file, you may have a hierarchy
  of settings like so:

      {
          \"development\": {
              \"rabbitmq\": {
                  \"host\": \"localhost\"
              }
          }
      }
          
  The \"host\" setting could the be retrieved by calling:

      (settings/for :rabbitmq.host)

  Don't worry about missing levels of the hierarchy.  If \"rabbitmq\" is not
  present, a nil value is returned by the (settings/for) function.

  By default we look for a configuration file called \"settings.json\" on the
  classpath.  You may override this by supplying a TANDEM_CONFIG environment
  variable with the full path of the configuration file, for example:

      TANDEM_CONFIG=/etc/tandem.json lein run

  To override the environment (\"development\" by default), set either the
  RING_ENV or the TANDEM_ENV environment variable."
  
  ;; TODO:  YAML
  
  (:refer-clojure :exclude (for load))
  (:use [clojure.data.json :only (read-json)]
        [clojure.java.io :only (resource reader)])
  (:require [clojure.string :as str]
            fs))

(defonce ^:dynamic *settings* nil)

;; Use either TANDEM_ENV or RING_ENV for the environment; defaults to "development"
(def environment (or (System/getenv "TANDEM_ENV")
                     (System/getenv "RING_ENV")
                     "development"))

;; Path to the settings/configuration files, e.g. TANDEM_CONFIG=/etc/tandem
(def tandem-config (or (System/getenv "TANDEM_CONFIG")
                       (resource "settings.json")))


(defn load-settings
  "Load the settings from the file defined by tandem-config."
  []
  (alter-var-root #'*settings* (fn [_]
                                 (let [json (read-json (reader tandem-config) false)]
                                   (get json environment)))))

(defn deploy
  "Copy the settings.json file from the classpath (presumably in this or another
  JAR file to the current resources directory.  Typically you call this after
  including Tandem or tandem/settings in a project."
  []
  (if-not (fs/exists? "resources")
    (fs/mkdir "resources"))
  (if-not (fs/exists? "resources/settings.json")
    (spit "resources/settings.json" (slurp (resource "settings.json")))))

(defn for
  "Look up a setting for the current environment, i.e. the runtime environment
  defined in the TANDEM_ENV or RING_ENV environment variable when the application
  in started.

  Settings supports multiple configuration files.  If the top-level 
  You can \"deep dive\" for values within objects in the JSON settings using
  dot notation.  For example, if you have settings like this:

    {
      \"rabbitmq\": {
        \"host\": \"localhost\"
      }
    }

  you can reference the RabbitMQ host by calling:

    (settings/for :rabbitmq.host)

  You may also include a default:

    (settings/for :rabbitmq.host \"localhost\")
  "
  ([settings property default]
    (if (and settings property)
      (let [[key rest] (str/split property #"\.", 2)
            value (settings key)]
        (if rest
          (recur value rest default)
          (or value default)))))

  ([property default] (for (or *settings* (load-settings)) (name property) default))
  ([property] (for property nil)))
