(ns tandem.settings

  "Parse the config.json configuration file and manage settings across the
  entire application.  Also understands the current state of the server, whether
  it is in development, test, or production mode."

  (:refer-clojure :exclude (for load))
  (:use [clojure.data.json :only (read-json)]
        [clojure.java.io :only (resource reader)])
  (:require [clojure.string :as str]))

(defonce ^:dynamic *settings* nil)

(def env (or (System/getenv "RING_ENV") "development"))

;; TODO:  Support an external configuration file, such as /etc/myapp/config.json

(defn load
  "Load the settings for the given environment and cache them.  Calling this
  again will reload the settings.  The environment defaults to 'development'."
  [& [environment]]
  (alter-var-root #'*settings* (fn [_]
                                 (let [e (or environment "development")
                                       json (read-json (reader (resource "config.json")) false)]
                                   (get json (name e))))))

(defn for
  "Look up a setting for the current environment, i.e. the runtime environment
  defined in the singularity.env system property.

  You can \"deep dive\" for values within objects in the JSON settings using
  dot notation.  For example, if you have settings like this:

    {
      \"rabbitmq\": {
        \"host\": \"localhost\"
      }
    }

  you can reference the RabbitMQ host by calling:

    (settings/for :rabbitmq.host)

  You may also include a default:

    (settings/for :rabbitmq.host \"localhost\")
  "
  ([settings property default]
    (if (and settings property)
      (let [[key rest] (str/split property #"\.", 2)
            value (settings key)]
        (if rest
          (recur value rest default)
          (or value default)))))

  ([property default] (for (or *settings* (load env)) (name property) default))
  ([property] (for property nil)))
