(ns tandem.mail

  "A Clojure wrapper around Apache's Commons Mail library."
  
  (:require [tandem.settings :as settings])
  (:import [org.apache.commons.mail SimpleEmail MultiPartEmail HtmlEmail EmailAttachment]))

(defn- setup
  "Add the basic information to the email object."
  [email {:keys [to from subject]}]
  (doto email
    (.setHostName (settings/for :mail.smtp.host "localhost"))
    (.setSmtpPort (settings/for :mail.smtp.port 25))
    (.setAuthentication (settings/for :mail.smtp.user) (settings/for :mail.smtp.password))
    (.setTLS (settings/for :mail.smtp.tls false))
    (.setFrom (or from (settings/for :mail.from) "root@localhost"))
    (.addTo to)
    (.setSubject subject)))

(defn- add-attachments
  "Associate any attachments to the email."
  [email attachments]
  (if attachments
    (doseq [attachment attachments] (.attach email attachment))))

(defn attachment
  "Create an attachment for a mail message."
  [{:keys [path description name]}]
  (doto (EmailAttachment.)
    (.setPath path)
    (.setDisposition EmailAttachment/ATTACHMENT)
    (.setDescription description)
    (.setName name)))

(defn simple
  "Send a simple text email."
  [{:keys [to from subject text] :as params}]
  (let [email (setup (SimpleEmail.) params)]
    (.setMsg email text)
    (.send email)))

(defn multipart
  [{:keys [to from subject text attachments] :as params}]
  (let [email (setup (MultiPartEmail.) params)]
    (if text (.setMsg email text))
    (add-attachments email attachments)
    (.send email)))

(defn html
  "Send an HTML mail message."
  [{:keys [to from subject html text attachments] :as params}]
  (let [email (setup (HtmlEmail.) params)]
    (if html (.setHtmlMsg email html))
    (if text (.setTextMsg email text))
    (add-attachments email attachments)
    (.send email)))
    
  ;; // embed the image and get the content id
  ;; URL url = new URL("http://www.apache.org/images/asf_logo_wide.gif");
  ;; String cid = email.embed(url, "Apache logo");
  
