(ns tandem.error-handling

  "A simple error system that throws exceptions to escape bad situations, and
  provides a Ring-compatible handler to deal with these errors."

  (:use ring.util.response)
  (:require [clojure.tools.logging :as log]
            [clojure.data.json :as json])
  (:import tandem.TandemException))

(def ^:const STATUS-VALUES
  {:bad-request 400
   :unauthorized 401
   :payment-required 402
   :forbidden 403
   :not-found 404
   :method-not-allowed 405
   :not-acceptable 406
   :proxy-authentication-required 407
   :request-timeout 408
   :conflict 409
   :gone 410
   :length-required 411
   :precondition-failed 412
   :request-entity-too-large 413
   :request-uri-too-long 414
   :unsupported-media-type 415
   :requested-range-not-satisfiable 416
   :expectation-failed 417
   :im-a-teapot 418
   :enhance-your-calm 420
   :unprocessable-entity 422
   :locked 423
   :failed-dependency 424
   :unordered-collection 425
   :upgrade-required 426
   :precondition-required 428
   :too-many-requests 429
   :request-header-fields-too-large 431
   :no-response 444
   :retry-with 449
   :block-by-windows-parental-controls 450
   :client-closed-request 499
   :internal-server-error 500
   :not-implemented 501
   :service-unavailable 503
   :gateway-timeout 504
   :http-version-not-supported 505
   :variant-also-negotiates 506
   :insufficient-storage 507
   :loop-detected 508
   :bandwidth-limit-exceeded 509
   :not-extended 510
   :network-authentication-required 511
   :network-read-timeout-error 598
   :network-connection-timeout-error 599}
 
(defn code-for
  "Convert a keyword status like :not-found to an HTTP response code like 404."
  [status]
  (get STATUS-VALUES status 500))

(defn status-for
  "Convert the HTTP error code to its corresponding status keyword.  The
  inverse of code-to-status (obviously).  See that function for details."
  [code]
  (first (first (filter #(= (last %) code) STATUS-VALUES))))

(defn failure
  "Throws a TandemException.  Makes it easy to generate error messages that
  bubble up to the API layer.  For example:

    (failure :not-found \"Could not find a user named %s.\" name)
  "
  [status message & substitutions]
  (let [code (code-for status)
        message (apply format (cons message substitutions))]
    (throw (TandemException. code message))))

(defn duplicate
  "Like failure, but used when a duplicate object is found."
  [existing message & substitutions]
  (let [code (code-for :duplicate)
        message (apply format (cons message substitutions))]
    (throw (TandemException. code message existing))))

;; TODO:  HTML and XML
(defn wrap-errors
  "A Ring handler for trapping exceptions raised in the application and returns
  either meaningful HTML, JSON, or XML formatted documents."
  [handler]
  (fn [request]
    (try
      (handler request)
      (catch TandemException e
        (let [s (.status e)
              d (.duplicate e)
              m (.getMessage e)
              res {:error m :status (status-for s) :code s}
              res (if d (assoc res :duplicate d) res)]
          (log/error e)
          (->
            res
            json/json-str
            response
            (content-type "application/json")
            (status s)))))))
           
