(ns tandem.server
  (:use ring.middleware.stacktrace
        ring.middleware.reload
        net.cgrand.enlive-html
        [tandem.http.headers :only [with-headers-from]])
  (:require [clojure.string :as str]
            [tandem.http.headers :as headers]
            [tandem.http.header :as header]
            [tandem.http.charset :as charset]
            [tandem.http.response :as response])
  (:import tandem.TandemException
           java.net.InetAddress
           java.net.InetSocketAddress
           java.util.concurrent.Executors
           org.jboss.netty.bootstrap.ServerBootstrap
           org.jboss.netty.channel.socket.nio.NioServerSocketChannelFactory
           [org.jboss.netty.channel
            Channels
            ChannelPipeline
            ChannelPipelineFactory
            SimpleChannelUpstreamHandler
            ChannelHandlerContext
            MessageEvent
            ExceptionEvent]
           [org.jboss.netty.buffer
            ChannelBufferInputStream]
           [org.jboss.netty.handler.codec.http
            QueryStringDecoder
            HttpContentCompressor
            HttpRequestDecoder
            HttpResponseEncoder
            HttpChunkAggregator
            HttpHeaders
            HttpRequest]))

(defonce ^:dynamic *server* (ref {}))

;; Used by bootstrap to create a Netty server instance
(defn- cached-thread-pool [] (Executors/newCachedThreadPool))
(defn- channel-factory [] (NioServerSocketChannelFactory. (cached-thread-pool) (cached-thread-pool)))

(deftemplate exception-page "500.html"
  [^Throwable exception]
  [:title] (content (.getName (class exception)))
  [:div.details :h1] (content (.getName (class exception)))
  [:div.details :p.message] (content (.getMessage exception))
  [:div.stacktrace :ul :li] (clone-for [element (.getStackTrace exception)]
                                   (content (str element))))

(defn- exception-status
  "Look up the HTTP response status for the given exception.  If not a
  TandemException, returns 500."
  [exception]
  (if (instance? TandemException exception)
    (.getStatus exception)
    500))
  
;; TODO:  Production vs. development/testing
;; TODO:  Logging?
;; TODO:  JSON, XML?
(defn display-error
  "Display the error (500) page when an exception is thrown."
  [^ChannelHandlerContext context ^ExceptionEvent event]
  (response/respond-with
    (let [cause (.getCause event)]
      (response/status (exception-status cause))
      (response/content-type "text/html")
      (response/for context (exception-page cause)))))
  
(defn- parse-uri
  "Parse out the query string parameters and the base path."
  [uri]
  (let [qsd (QueryStringDecoder. uri)]
    {:path (.getPath qsd)
     :params (.getParameters qsd)}))

;; TODO: params, cookies, querystring
(defn- parse-request
  "Convert the Netty details into something that Ring can handle.  We include
  not only the required basics, but both query string and body parameters,
  and the 'Accept' values."
  [context event]
  (let [message (.getMessage event)
        channel (.getChannel context)
        server-address (.getLocalAddress channel)
        uri (.getUri message)
        {path :path qs-params :params} (parse-uri uri)]
    (with-headers-from message
      {:server-name (-> server-address .getAddress .getHostName)
       :server-port (-> server-address .getPort)
       :remote-addr (-> channel .getRemoteAddress .getAddress .getHostAddress)
       :uri uri
       :path path
       :scheme (keyword (headers/get header/x-scheme "http"))
       :request-method (-> message .getMethod .getName .toLowerCase keyword)
       :headers headers/*headers*
       :content-type (headers/content-type)
       :content-length (HttpHeaders/getContentLength message)
       :character-encoding (headers/character-encoding)
       :accept (headers/accept)
       :body (ChannelBufferInputStream. (.getContent message))
       :keep-alive (HttpHeaders/isKeepAlive message)})))

;; TODO:  Support SSL!
(defn- pipeline-factory
  [^SimpleChannelUpstreamHandler handler]
  (reify ChannelPipelineFactory
    (getPipeline [this] (doto (Channels/pipeline)
                          (.addLast "decoder" (HttpRequestDecoder.))
                          (.addLast "aggregator" (HttpChunkAggregator. 1048576))
                          (.addLast "encoder" (HttpResponseEncoder.))
                          (.addLast "deflater" (HttpContentCompressor.))
                          (.addLast "handler" handler)))))

(defn- inet-socket-address
  "Create an InetSocketAddress based on the given address and port information.
  Understands both IPv4 addresses (for example, 127.0.0.1) and IPv6 addresses
  (such as 2001:db8:85a3:0:0:8a2e:370:7334).  If the address is nil, creates
  a socket address that binds to all IPs on the machine."
  [address port]
  (let [address (if address (InetAddress/getByName address) nil)]
    (InetSocketAddress. address port)))

(defn- create-handler
  "Create a default handler for a Netty server instance."
  [app]
  (proxy [SimpleChannelUpstreamHandler] []

    (messageReceived [^ChannelHandlerContext context ^MessageEvent event]
      (let [req (parse-request context event)
            res (app req)]
        (if res (response/for context res))))
    
    (exceptionCaught [^ChannelHandlerContext context ^ExceptionEvent event]
      (display-error context event))))

(defn- dev-handler?
  "Should we wrap this as a dev handler?"
  [app environment]
  (if (= environment "development")
    (reduce
     (fn [h m] (m h))
     app
     [wrap-stacktrace
      wrap-reload])
    app))

(defn- bootstrap
  "Create a ServerBootstrap for an HTTP server.  Returns the Channel for the
  bootstrap service (which can be used to stop the server)."
  [app environment ip port]
  (let [server (ServerBootstrap. (channel-factory))]
    (.setPipelineFactory server (pipeline-factory (create-handler (dev-handler? app environment))))
    (.bind server (inet-socket-address ip port))))

(defn- start-server
  "Start a server and record that it's running (so we can stop it layer, duh)."
  [server {:keys [app ip port environment on-boot on-shutdown] :as options}]
  (if (:server server)
    (throw (Exception. (str "A " environment " server is already running on port " port)))
    (do
      (if on-boot (on-boot))
      {:server (bootstrap app environment ip port)
       :environment environment
       :ip ip
       :port port
       :on-shutdown on-shutdown})))

(defn- stop-server
  "Stop the running server."
  [{:keys [server port on-shutdown] :as existing}]
  (when server
    (.disconnect server)
    (.close server)
    (if on-shutdown (on-shutdown)))
  {})
    
(defn stop
  "Stop the running instance of the Netty server."
  []
  (dosync (alter *server* stop-server)))

(defn start
  "Start an instance of the Netty server.  The following options are available:

    ip          - configure the IP address to listen to on this machine;
                  defaults to all 
    port        - the port to listen on; defaults to 8080
    on-boot     - points to a function to call before the server starts
    on-shutdown - points to a function to call when the server stops
    environment - the deployment environment; defaults to 'development'; may be
                  set directly, using the 'tandem.env' system property, or via
                  the RING_ENV environment variable
  "
  [app & [{:keys [ip port on-boot on-shutdown environment] :as options}]]
  (let [port (or port 8080)
        environment (or environment
                        (System/getProperty "tandem.env" (System/getenv "RING_ENV"))
                        "development")]
    (dosync (alter *server* start-server (assoc options :app app :port port :environment environment)))))

  