(ns tandem.http.response

  "Build a response object to send back across the Netty Channel.  You shouldn't
  have to use any of these functions directly.  Instead, use the Ring response
  handler, and let the tandem.server deal with the Netty response.

  The failure function in this namespace can be useful to immediately halt the
  workflow and send an error message to the user."

  (:refer-clojure :exclude [send for])
  (:use tandem.http.charset
        [clojure.data.json :only [json-str]])
  (:require [tandem.http.header :as h]
            [tandem.http.status :as s])
  (:import tandem.TandemException
           java.net.URLConnection
           java.io.RandomAccessFile
           org.jboss.netty.channel.ChannelFutureListener
           org.jboss.netty.channel.DefaultFileRegion
           org.jboss.netty.buffer.ChannelBuffers
           org.jboss.netty.handler.stream.ChunkedStream
           org.jboss.netty.handler.stream.ChunkedFile
           org.jboss.netty.handler.ssl.SslHandler
           [org.jboss.netty.handler.codec.http
            Cookie
            CookieDecoder
            CookieEncoder
            DefaultHttpResponse
            HttpResponseStatus
            HttpHeaders
            HttpVersion]))

(def ^:dynamic *response* nil)

(def ^:static http-1-0 HttpVersion/HTTP_1_0)
(def ^:static http-1-1 HttpVersion/HTTP_1_1)

(defn failure
  "Raise a TandemException, which will halt execution and fire up through the
  handlers to be dealt with by the exception handling output layer."
  [status message]
  (throw (TandemException. message status)))

(defn create
  "Create a Netty HttpResponse object.  Call this, then apply any header, status
  and content to the response before sending.  Defaults to HTTP 1.1 OK."
  []
  (DefaultHttpResponse. http-1-1 s/ok))

(defn header
  "Set a header value in the response, such as Content-Type.  See
  tandem.http.header for valid name values."
  ([response name value] (if (and name value) (doto response (.setHeader name value))))
  ([name value] (header *response* name value)))

(defn status
  "Set the HTTP response status code.  See tandem.http.status for valid status
  codes."
  ([response code] (if code (.setStatus response (HttpResponseStatus/valueOf code))))
  ([code] (status *response* code)))

(defn content-type
  "Shortcut to set the content type (text/html, application/json, etc) and
  character set.  Defaults to UTF-8."
  ([response mime-type charset] (header response h/content-type (str mime-type "; charset=" (or charset "UTF-8"))))
  ([mime-type charset] (content-type *response* mime-type charset))
  ([mime-type] (content-type *response* mime-type nil)))

(defn content-type-from-file
  "Guess at the mime-type of the file based on its extension, e.g. .jpg,
  .html, etc."
  ([response file]
     (let [mime-type (URLConnection/guessContentTypeFromName (str file))]
       (header response h/content-type mime-type)))
  ([file] (content-type-from-file *response* file)))

(defn- buffer
  "Copy the content into the channel buffer.  Defaults to UTF-8."
  [content & [charset]]
  (ChannelBuffers/copiedBuffer (str content) (or charset utf-8)))

(defn content
  "Set the content to return.  Defaults to UTF-8."
  ([response data charset] (doto response (.setContent (buffer data charset))))
  ([data charset] (content *response* data charset))
  ([data] (content *response* data nil)))

(defn ssl?
  "Is the given channel an SSL channel?"
  [channel]
  (let [pipeline (.getPipeline channel)]
    (not (nil? (.get pipeline (class SslHandler))))))

(defn- auto-close
  "Used to automatically close the future when finished writing, i.e. when
  Keep-Alive is not enabled."
  [channel-future]
  (.addListener channel-future ChannelFutureListener/CLOSE))

;; TODO:  include cookies in the response?
(defn send-response
  "Send the response to the event channel.  Returns the ChannelFuture."
  ([response channel]
     (let [keep-alive (:keep-alive response)]
       (if keep-alive (header response :content-length (-> response .getContent .readableBytes)))
       (let [channel-future (.write channel response)]
         (if-not keep-alive (auto-close channel-future))
         channel-future)))
  ([channel] (send-response *response* channel)))

(defn send-file
  "Send a file to the event channel."
  ([response channel file]
     (content-type-from-file response file)
     (let [keep-alive (:keep-alive response)
           file (RandomAccessFile. file "r")
           length (.length file)]
       
       (header response h/content-length length)
       (.write channel response)

       (let [future (if (ssl? channel)
                      (.write channel (ChunkedFile. file 0 length 8192))
                      (let [region (DefaultFileRegion. (.getChannel file) 0 length)
                            future (.write channel region)]
                        (.addListener future (reify ChannelFutureListener
                                               (operationComplete [this future]
                                                 (.releaseExternalResources region))))
                        future))]
         (if-not keep-alive (auto-close future))
         future)))
  ([channel file] (send-file *response* channel file)))

(defmacro respond-with
  "Wrap a series of forms with a single response."
  [& forms]
  `(binding [*response* (create)]
     ~@forms))

(defprotocol RingResponse
  (send [this channel]))

(extend-type java.lang.String
  RingResponse
  (send [this channel]
    (content this)
    (send-response channel)))

(extend-type clojure.lang.ISeq
  RingResponse
  (send [this channel]
    (content (apply str this))
    (send-response channel)))

(extend-type java.util.Map
  RingResponse
  (send [this channel]
    (content-type "application/json")
    (content (json-str this))
    (send-response channel)))

(extend-type java.io.InputStream
  RingResponse
  (send [this channel]
    (.write channel *response*)
    (auto-close (.write channel (ChunkedStream. this)))))

(extend-type java.io.File
  RingResponse
  (send [this channel]
    (send-file channel this)))

(extend-type nil
  RingResponse
  (send [this channel]
    nil))

(defn for
  "Generate a response for Netty from a Ring response map.  See the Ring docs
  at https://github.com/mmcgrana/ring/blob/master/SPEC for details about what
  is expected in the response map."
  [context & [res]]
  (respond-with
    (let [channel (.getChannel context)]
      (status (:status res))
      (doseq [[name value] (:headers res)] (header name value))
      (send (:body res) channel))))
