(ns tandem.couch.session.store

  (:use ring.middleware.session.store
        [clj-time.core :only [after? plus now minutes]]
        [clj-time.format :only [formatters unparse]]
        tandem.misc)
  (:require [clojure.tools.logging :as log]
            [tandem.couch :as couch]))

(def ^{:private true} json-format (formatters :date-time-no-ms))

(defn- past-due?
  "Is this given document expired?"
  [{expires :expires :as doc}]
  (if expires (after? (now) expires)))

(defn- expiration-date
  "Calculate when the document should expire."
  [expires-in]
  (unparse json-format (plus (now) (minutes expires-in))))

(deftype CouchStore [db expires]
  SessionStore
  (read-session [_ key]
    (couch/with db
                (let [value (or (couch/get key) {})
                      expiration (:expiration value)]
                  (if (past-due? expiration)
                    (couch/delete value)
                    value))))

  (write-session [_ key data]
    (couch/with db
                (let [{:keys [_id _rev]} (couch/get key)]
                  (let [data (assoc-if data :_id key :_rev _rev :expiration (expiration-date expires))]
                    (let [{_id :_id} (couch/put data)]
                      _id)))))
                
  (delete-session [_ key]
    (couch/with db (couch/delete key))
    nil))

(defn couch-store
  "Create a CouchDB session store object.  The expires time is 20 minutes by
  default, and the database core name is 'sessions'."
  [& {:keys [db expires]}]
  (let [db (or db "sessions")
        expires (or expires 20)]
    (couch/create-database db)
    (CouchStore. db expires)))
