(ns tandem.couch

  "A CouchDB library for Clojure, built on the clj-apache-http library."

  (:refer-clojure :exclude (get find))
  (:use [clojure.java.io :only (resource as-file)]
        [clojure.data.json :only (json-str read-json)]
        tandem.misc)
  (:require [clojure.string :as str]
            [com.twinql.clojure.http :as http]
            [tandem.settings :as settings])
  (:import org.apache.http.entity.StringEntity
           java.net.URLEncoder))

;; TODO:  Attachments to documents

;; Declare the current database using the "with" function 
(def ^:dynamic *db* nil)
(def ^:dynamic *base-name* nil)

;; A connection manager for clj-apache-http, so we can pool connections
(def cm (doto (http/thread-safe-connection-manager)
          (.setDefaultMaxPerRoute (or (settings/for :couch.connection.max-per-route) 5))
          (.setMaxTotal (or (settings/for :couch.connection.max-total) 25))))

;; Remember the CouchDB server URL, so we don't have to constantly look it up
(def url (settings/for :couch.url))
(def app (settings/for :couch.app))

;; TODO:  I'm cheating a bit with these...just a simple regex, instead of a proper parser
(def map-rx #"(?ms)function\(\s*\w+\s*\).*?^}")
(def reduce-rx #"(?ms)function\(\s*\w+\s*,\s*\w+\s*,\s*\w+\s*\).*?^}")

;; Convert the filename into the name of the map/reduce view
(def view-rx #"(?i)^(.*)(?:.js)$")

(defn- encode-path
  [s]
  (if s
    (->
     (java.net.URLEncoder/encode (name s) "UTF-8")
     (str/replace "+" "%20")
     (str/replace "%2F" "/"))))

(defn to
  "Create a full URL to the given path resource.  For example,

    (to \"p\" \"some-slug\" \"m\")

  would return a URL that looks like:

    http://localhost:8080/p/some-slug/m
  "
  [& parts]
  (str url "/" (str/join "/" (map encode-path parts))))

(defn json-entity
  "Convert an object (map) into a JSON HttpEntity for clj-apache-http."
  [obj]
  (StringEntity. (json-str obj) "application/json" "UTF-8"))

(defn make-request
  "Make the HTTP request to the CouchDB server.  It returns a map with :status,
  :error, and :results keys.

  If you include a body, it will be converted to JSON, since that's all Couch
  understands or accepts."
  [http-func url params & [body]]
  (let [body (if body (json-entity body))
        {status :code reason :reason results :content} (http-func url :query params :body body
                                                                  :as :json :connection-manager cm)]
    (if (< status 300)
      {:status status :results results}
      {:status status :error reason})))

(defn- http-get [url & [params]] (make-request http/get url params))
(defn- http-post [url & [params body]] (make-request http/post url params body))
(defn- http-put [url & [params body]] (make-request http/put url params body))
(defn- http-delete [url & [params]] (make-request http/delete url params))

(defmacro rescue-json
  "Like the rescue macro, but if the form returns an error JSON response, we
  return the given value instead."
  [form & [value]]
  `(let [response# ~form]
     (if (and
          (contains? response# :status)
          (contains? response# :error))
       ~value
       response#)))

(defn ok?
  "Did CouchDB return a 200/Ok response?"
  [results]
  (< (:status results) 300))

(defn databases
  "Get the list of databases in CouchDB.  Returns a future in response."
  []
  (http-get (to "_all_dbs")))

(defn database
  "Get the details about a database."
  [database-name]
  (http-get (to database-name)))

(defn named
  "Builds the name of a database to use with Couch to isolate it from other
  applications, even different environments of this application.  The name
  will be the app-database-env, e.g. 'tandem-users-development'."
  [database-name]
  (str/join "-" [app database-name settings/env]))
  
(defn create-database
  "Create a new database in Couch.  Returns the information about the database
  when it is created, i.e. it performs a database-info call.

  If the database already exists, Couch will return an error message.  This
  function ignores that message, and simply returns the details about the
  existing database."
  ([database-name]
     (let [database-name (named database-name)
           {status :status error :error results :results :as response} (http-put (to database-name))]
       (if (or (= status 201) (= status 412))
         (http-get (to database-name))
         response)))
  ([] (create-database *db*)))

(defn delete-database
  "Delete the given database.  Returns a 404 status if the database doesn't
  exist."
  ([database-name] (http-delete (to (named database-name))))
  ([] (delete-database *db*)))

(defmacro with
  "Work within the confines of the given database.  Make sure you create the
  database first.  For example:

    (with \"sample_database\"
      (add {:name \"Some Entry\" :city \"Some Place\"}))"
  [base-name & body]
  `(binding [*base-name* ~base-name
             *db* (named ~base-name)]
     ~@body))

(defn put
  "Create or update a document in the database.  If the document has an _id
  property, will use that.  If not, an ID will be created for the document.

  When updating a document, the _rev property is required.  Otherwise a 409
  Conflict exception will occur."
  [doc & {:keys [id]}]
  (if doc
    (let [id (or id (:_id doc))
          {{_id :id _rev :rev} :results} (if id
                                           (http-put (to *db* id) nil doc)
                                           (http-post (to *db*) nil doc))]
      (assoc doc :_id _id :_rev _rev))))

(defn get
  "Get a document from the database by its ID."
  [id]
  (if id
    (let [{status :status results :results} (http-get (to *db* id))]
      (if (= status 200) results))))

(defn delete
  "Delete a document from the database.  If you pass in the document to delete,
  the revision will be verified.  If you submit only the document ID, such as
  (delete \"abc123\"), then the document will be removed regardless of its
  revision (this is a moderately dangerous thing to do, by the way)." 
  [doc]
  (if doc
    (let [{:keys [_id _rev]} (if (map? doc)
                               doc
                               (get doc))]
      (http-delete (to *db* _id) {:rev _rev}))))
  
(defn bulk-update
  "Bulk create or update a number of Couch documents.  Wrap in a with function
  so we know the database.  If documents don't have _id values, the _id will be
  generated."
  [docs]
  (http-post (to *db* "_bulk_docs") nil {:docs docs}))

(defn- fix-keys
  "When submitting key values to a view, the value must be valid JSON.  The
  view function will call this function to correct any values that aren't
  JSON, e.g. convert strings to JSON-encoded strings."
  [options]
  (into {} (map (fn [[k v]]
                  (if (or (= k :key)
                          (= k :keys)
                          (= k :startkey)
                          (= k :endkey))
                    [k (json-str v)]
                    [k v])) options)))

(defn view
  "Return the results from running a view on the database.  The options are
  those defined in the CouchDB API:  key, startkey, endkey, limit, skip,
  descending, etc."
  ([document-name view-name options]
     (http-get (to *db* "_design" document-name "_view" view-name) (fix-keys options)))
  ([view-name options]
     (if (string? options)
       (view view-name options nil)
       (view *base-name* view-name options)))
  ([view-name] (view *base-name* view-name nil)))

(defn- parse-map-reduce
  "Parse a map/reduce JavaScript file for Couch and return it as a map for
  inclusion in the design document."
  [database-name filename]
  (let [contents (slurp (resource (str "couch/" database-name "/" filename)))
        view (keyword (last (re-find view-rx filename)))
        map-js (re-find map-rx contents)
        reduce-js (re-find reduce-rx contents)]

    (if map-js
      (let [document {view {:map map-js}}]
        (if reduce-js
          (assoc document :reduce reduce-js)
          document)))))
  
(defn design-document
  "Look in the resources/couch/document-name directory for a series of
  JavaScript files.  These should contain the map/reduce views for the database
  (and optionally the validation document (TODO)).  Wrap in a with macro to use
  the base-name of the database as the document name; or you can specify it.

    (couch/with \"some-database\"
      (couch/design-document)                   ; _id is _design/some-database
      (couch/design-document \"another-doc\"))  ; _id is _design/another-doc

  You need to save the document yourself.
  "
  [& [document-name]]
  (let [document-name (or document-name *base-name*)
        files (map str (.list (as-file (resource (str "couch/" document-name)))))]
    {:_id (str "_design/" document-name)
     :language "javascript"
     :views (reduce (fn [d f] (merge d (parse-map-reduce document-name f))) {} files)}))

  