(ns tailrecursion.javastar
  (:require
   [alandipert.interpol8 :refer [interpolating]]
   [clojure.core.cache :as cache])
  (:import
   [javax.tools
    DiagnosticCollector
    ForwardingJavaFileManager
    JavaCompiler
    JavaFileObject$Kind
    SimpleJavaFileObject
    StandardJavaFileManager
    ToolProvider]))

(defn source-object
  "Returns a JavaFileObject to store a class file's source."
  [class-name source]
  (proxy [SimpleJavaFileObject]
      [(java.net.URI/create (str "string:///"
                                 (.replace ^String class-name \. \/)
                                 (. JavaFileObject$Kind/SOURCE extension)))
       JavaFileObject$Kind/SOURCE]
    (getCharContent [_] source)))

(defn class-object
  "Returns a JavaFileObject to store a class file's bytecode."
  [class-name baos]
  (proxy [SimpleJavaFileObject]
      [(java.net.URI/create (str "string:///"
                                 (.replace ^String class-name \. \/)
                                 (. JavaFileObject$Kind/CLASS extension)))
       JavaFileObject$Kind/CLASS]
    (openOutputStream [] baos)))

(defn class-manager
  "An in-memory JavaFileManager for storing generated bytecode."
  [manager]
  (let [baos (java.io.ByteArrayOutputStream.)]
    (proxy [ForwardingJavaFileManager] [manager]
      (getClassLoader [location]
        (proxy [clojure.lang.DynamicClassLoader] []
          (findClass [name]
            (let [^clojure.lang.DynamicClassLoader this this]
              (proxy-super defineClass name (.toByteArray baos) nil)))))
      (getJavaFileForOutput [location class-name kind sibling]
        (class-object class-name baos)))))

(defn compile-java
  "Compiles the class-name implemented by source and loads it,
  returning the loaded Class."
  [class-name source]
  (let [compiler (ToolProvider/getSystemJavaCompiler)
        diag (DiagnosticCollector.)
        mgr (class-manager (.getStandardFileManager compiler nil nil nil))
        task (.getTask compiler nil mgr diag nil nil [(source-object class-name source)])]
    (if (.call task)

      (.loadClass (.getClassLoader ^StandardJavaFileManager mgr nil) class-name)
      (throw (RuntimeException.
              (str "java* compilation error: " (first (.getDiagnostics diag)) "\n"
                   "source of generated class: \n" source "\n"))))))

(defn occurrences
  "Count of the occurrences of substring in s."
  ([s substring]
     (occurrences 0 s substring))
  ([n ^String s ^String substring]
     (let [i (.indexOf s substring)]
       (if (neg? i)
         n
         (recur (inc n) (.substring s (inc i)) substring)))))

(defn substitute
  "Replace pattern in s with substitutions."
  [s pattern substitutions]
  (reduce #(.replaceFirst ^String %1 pattern %2) s substitutions))

(defn ^Class generate-class
  "Generates, compiles, and loads a class with a single method, m,
  containing the code string.  Defines m's signature using
  return-type, arg-types, and ~{} occurrences in source as with js*.

  Returns the loaded class's name as a symbol."
  [imports return-type arg-types code]
  (let [class-name (str (gensym "generated_class"))
        n (occurrences code "~{}")
        arg-names (mapv str (repeatedly n gensym))
        arguments (->> (map #(str %1 " " %2) arg-types arg-names)
                       (interpose \,)
                       (apply str))
        method-body (substitute code "~\\{\\}" arg-names)
        prelude (apply str (map #(format "import %s;\n" %) imports))
        class-body (interpolating
                    "package tailrecursion.java_STAR_;
                     #{prelude}
                     public class #{class-name} {
                       public static #{return-type} m (#{arguments}) {
                         #{method-body}
                       }
                     }")]
   (compile-java (str "tailrecursion.java_STAR_." class-name) class-body)))

(def prims-classes
  "Map of primitive aliases to Classes."
  '{boolean  Boolean/TYPE
    byte     Byte/TYPE
    char     Character/TYPE
    float    Float/TYPE
    int      Integer/TYPE
    double   Double/TYPE
    long     Long/TYPE
    short    Short/TYPE
    booleans (Class/forName "[Z")
    bytes    (Class/forName "[B")
    chars    (Class/forName "[C")
    floats   (Class/forName "[F")
    ints     (Class/forName "[I")
    doubles  (Class/forName "[D")
    longs    (Class/forName "[J")
    shorts   (Class/forName "[S")})

(def prims-strings
  "Type aliases for use with the return-type and arg-types arguments
   of java*."
  '{void          "void"
    boolean       "boolean"
    byte          "byte"
    char          "char"
    float         "float"
    int           "int"
    double        "double"
    long          "long"
    short         "short"
    booleans      "boolean[]"
    bytes         "byte[]"
    chars         "char[]"
    floats        "float[]"
    ints          "int[]"
    doubles       "double[]"
    longs         "long[]"
    shorts        "short[]"})

(def classes-strs
  "Map of primitive classes to type strings."
  {Boolean/TYPE         "boolean"
   Byte/TYPE            "byte"
   Character/TYPE       "char"
   Float/TYPE           "float"
   Integer/TYPE         "int"
   Double/TYPE          "double"
   Long/TYPE            "long"
   Short/TYPE           "short"
   (Class/forName "[Z") "boolean[]"
   (Class/forName "[B") "byte[]"
   (Class/forName "[C") "char[]"
   (Class/forName "[F") "float[]"
   (Class/forName "[I") "int[]"
   (Class/forName "[D") "double[]"
   (Class/forName "[J") "long[]"
   (Class/forName "[S") "short[]"})

(def method-cache
  "Cache of method signatures to Methods.  Used when java* is AOT'd."
  (atom (cache/lu-cache-factory {} :threshold 1024)))

(defn ^java.lang.reflect.Method generate-method
  "Generates a method satisfying arguments, returning a matching
  method if one has already been generated."
  [imports return-class arg-classes code]
  (let [k [return-class arg-classes code]]
    (if-let [meth (get @method-cache k)]
      (do (swap! method-cache cache/hit k) meth)
      (let [imports-str (mapv #(.getName ^Class %) imports)
            return-str (or (classes-strs return-class)
                           (.getName ^Class return-class))
            arg-strs (mapv #(or (classes-strs %)
                                (.getName ^Class %)) arg-classes)
            klass (generate-class imports-str return-str arg-strs code)
            meth (.getMethod klass "m" (into-array Class arg-classes))]
        (do (swap! method-cache cache/miss k meth) meth)))))

(defmacro java*
  "Similar to ClojureScript's js*.  Compiles a Java code string into a
  Java method and invokes the method with args.

  java* has more arguments than js*.  imports is a vector of
  zero or more class names.  return-type and arg-types
  may be either Java classes or symbol aliases for primitive types and
  arrays.  See prims-classes keys for available aliases.

  Example:

  (def java-add #(java* [] long [long long] \"return ~{} + ~{};\" %1 %2))
  (java-add 1 2) ;=> 3"
  [imports return-type arg-types code & args]
  {:pre [(= (count arg-types) (count args))]}
  `(let [meth# (generate-method
                ~imports
                ~(or (prims-classes return-type) return-type)
                ~(mapv #(or (prims-classes %) %) arg-types)
                ~code)]
     (.invoke meth# nil (object-array [~@args]))))
