;; Copyright (c) Alan Dipert and Micha Niskin. All rights reserved.
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file epl-v10.html at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns tailrecursion.castra
  (:require [cognitect.transit :as t]))

;; helpers ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- safe-pop
  [x]
  (or (try (pop x) (catch js/Error e)) x))

;; public api ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def ^:dynamic *validate-only*
  "Only validate request parameters, don't actually do it?"
  nil)

(defn ex?
  "Returns true if x is an ExceptionInfo."
  [x]
  (instance? ExceptionInfo x))

(defn make-ex
  "Given either an existing exception or a map, returns an ExceptionInfo
  object with the special status and serverStack properties set. If ex is
  an exception already then ex itself is returned."
  [ex]
  (if (ex? ex)
    ex
    (let [{:keys [status message data stack cause]} ex]
      (doto (ex-info message data cause)
        (aset "serverStack" stack)
        (aset "status" status)))))

(defn ajax-fn
  "Ajax request implementation using the standard jQuery ajax machinery."
  [{:keys [url json->clj timeout credentials]} data headers done fail always]
  (let [opts {"async"       true
              "contentType" "application/json"
              "data"        data
              "dataType"    "text"
              "headers"     headers
              "processData" false
              "type"        "POST"
              "url"         url
              "timeout"     timeout
              "xhrFields"   {"withCredentials" credentials}}
        ex    #(make-ex {:message "Server Error" :cause %})]
    (-> js/jQuery
        (.ajax (clj->js opts))
        (.done (fn [_ _ x]
                 (done (try (json->clj (aget x "responseText"))
                            (catch js/Error e (ex e))))))
        (.fail (fn [x t r]
                 (let [status  (.-status x)
                       head    (aget x "getResponseHeader")
                       tunnel? (.call head x "X-Castra-Tunnel")
                       body    (if-not tunnel?
                                 {:message r :status status}
                                 (try (json->clj (aget x "responseText"))
                                      (catch js/Error e (ex e))))]
                   (fail (make-ex body)))))
        (.always (fn [_ _] (always))))))

(defn- ajax
  [{:keys [ajax-fn clj->json] :as opts} expr done fail always]
  (let [headers {"X-Castra-Csrf"          "true"
                 "X-Castra-Tunnel"        "transit"
                 "X-Castra-Validate-Only" (str (boolean *validate-only*))
                 "Accept"                 "application/json"}
        expr    (if (string? expr) expr (clj->json expr))
        done    #((if (ex? %) fail done) %)]
    (ajax-fn opts expr headers done fail always)))

(defn with-default-opts
  [& [opts]]
  (->> opts (merge {:timeout     0
                    :credentials true
                    :ajax-fn     ajax-fn
                    :json->clj   (partial t/read (t/reader :json))
                    :clj->json   (partial t/write (t/writer :json))
                    :url         (.. js/window -location -href)})))

(defn mkremote
  "Given state error and loading input cells, returns an RPC function. The
  optional :url keyword argument can be used to specify the URL to which the
  POST requests will be made."
  [endpoint state error loading & [opts]]
  (fn [& args]
    (let [p (.Deferred js/jQuery)]
      (swap! loading (fnil conj []) p)
      (let [xhr (ajax
                  (with-default-opts opts)
                  `[~endpoint ~@args]
                  #(do (reset! error nil) (reset! state %) (.resolve p %))
                  #(do (reset! error %) (.reject p %))
                  #(swap! loading (fn [x] (vec (remove (partial = p) x)))))]
        (doto p (aset "xhr" xhr))))))
