(ns tailrecursion.boot.task.cljs
  (:require
    [cljs.closure                   :as cljs]
    [clojure.string                 :refer [split join]]
    [clojure.java.io                :refer [file delete-file make-parents]]
    [tailrecursion.boot.deps        :refer [deps]]  
    [tailrecursion.boot.tmpregistry :refer [mk! mkdir!]]))

;; INTERNAL ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defrecord CljsSourcePaths [paths]
  cljs/Compilable
  (-compile [this opts]
    (mapcat #(cljs/-compile % opts) paths)))

(let [last-counter (atom 0)]
  (def cljs-counter! #(swap! last-counter inc)))

(defn file? [f] (when (try (.isFile (file f)) (catch Throwable _)) f))

(defn clean! [& files]
  (doseq [f files]
    (doall (->> f file file-seq (keep file?) (map #(delete-file % true))))))

(defn install-cljs-deps! [src-paths depjars incs exts libs flibs]
  (let [name*     #(.getName (file %))
        match     #(last (re-find #"[^.]+\.([^.]+)\.js$" %))
        dirmap    {"inc" incs "ext" exts "lib" libs "flib" flibs}
        outfile   #(file %1 (str (format "%010d" (cljs-counter!)) "_" (name* %2)))
        write1    #(when-let [d (dirmap (match %1))]
                     (spit (doto (outfile d %1) make-parents) (slurp %2))) 
        write     #(map (partial apply write1) %)
        path-seq  (fn [x] (map #(.getPath %) (file-seq (file x))))
        dep-files (->> depjars (map second) (mapcat identity))
        src-files (->> src-paths (mapcat path-seq) (keep file?))]
    (doall (->> dep-files reverse write))
    (doall (->> src-files sort (map (juxt identity file)) write))))

;; PUBLIC ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn wrap-cljs [continue src-paths depjars flib-out lib-out ext-out inc-out opts]
  (assert (:output-to opts) "No :output-to option specified.")
  (fn [event]
    (clean! (:output-to opts) flib-out lib-out ext-out inc-out)
    (install-cljs-deps! src-paths depjars inc-out ext-out lib-out flib-out) 
    (let [{:keys [output-to]} opts
          file? #(.isFile %)
          path* #(.getPath %)
          files #(filter file? (file-seq %))
          paths #(mapv path* (files %))
          cat   #(join "\n" (mapv slurp %)) 
          srcs  (CljsSourcePaths. src-paths)
          exts  (paths ext-out)
          incs  (cat (sort (files inc-out)))]
      (cljs/build srcs (update-in opts [:externs] into exts))
      (spit output-to (str incs "\n" (slurp output-to)))
      (continue event))))

(defn cljs [boot output-to & [opts]]
  (let [base-opts   {:output-dir    nil
                     :optimizations :whitespace
                     :warnings      true
                     :externs       []
                     :libs          []
                     :foreign-libs  []
                     :pretty-print  true}
        src-paths   (:directories @boot)
        tmp         (get-in @boot [:system :tmpregistry])
        depjars     (deps boot)
        output-dir  (mkdir! tmp ::output-dir)
        flib-out    (mkdir! tmp ::flib-out)
        lib-out     (mkdir! tmp ::lib-out)
        ext-out     (mkdir! tmp ::ext-out)
        inc-out     (mkdir! tmp ::inc-out)
        x-opts      (->> {:output-to output-to, :output-dir output-dir}
                      (merge base-opts opts))]
    #(wrap-cljs % src-paths depjars flib-out lib-out ext-out inc-out x-opts)))
