(ns t6.graph-transform.viz
  (:require [clojure.set :as set]
            [rhizome.dot :as dot]
            [rhizome.viz :as viz]))

(defn ->str
  [x]
  {:label (str x)})

(defn graph->dot
  [graph graph-descriptor]
  (apply dot/graph->dot
         (:nodes graph)
         (fn [node]
           (mapv (fn [[_ x]] x) (-> graph :edges (get node))))
         :node->cluster (fn [_] (:cluster-id graph-descriptor))
         :node->descriptor
         (:node->descriptor graph-descriptor
                            (fn [node] {:label (str node)}))

         :edge->descriptor
         (:edge->descriptor graph-descriptor
                            (fn [src dest]
                              (let [label (some (fn [[l d]] (if (= d dest) l))
                                                (-> graph :edges (get src)))]
                                {:label (str label)})))
         (flatten graph-descriptor)))

(defn view-graph
  [graph graph-descriptor]
  (viz/view-image (viz/dot->image (graph->dot graph graph-descriptor))))

(defn graphs->dot
  [graphs]
  (let [;; number every graph and add graph number to nodes
        graphs (vec (map-indexed (fn [i g]
                                   (assoc g
                                     :nodes (mapv (fn [node] [i node])
                                                  (:nodes g))))
                                 graphs))]
    (dot/graph->dot
     ;; nodes
     (reduce set/union (map :nodes graphs))

     ;; adjacent
     (fn [[i node]]
       (let [graph (nth graphs i)]
         (mapv (fn [[_ x]] [i x]) (-> graph :edges (get node)))))

     :node->cluster
     (fn [[i node]] i)

     :node->descriptor
     (fn [[i node]] {:label (str node)})

     :edge->descriptor
     (fn [[i src] [j dest]]
       (let [graph (nth graphs i)
             label (some (fn [[l d]] (if (= d dest) l))
                         (-> graph :edges (get src)))]
         {:label (str label)})))))

(defn view-graphs
  [graphs]
  (viz/view-image (viz/dot->image (graphs->dot graphs))))

(defn view-diff
  "Runs `f` on graph with the given arguments and shows the graphs
  before and after running `f` for a side by side comparison."
  [f graph & args]
  (let [updated-graph (apply f graph args)]
    (view-graphs [graph updated-graph])))
