(ns t6.graph-transform.core
  (:require [clojure.set :as set]
            [t6.graph-transform.util :as u]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Schemas
;;; =======

(def AdjacentEdges
  (u/if-schema
   {(s/named s/Any "parent node")
    #{[(s/one s/Any "edge label or properties")
       (s/one s/Any "child node")]}}))

;; `core.logic` can't unify with sets
(def AdjacentEdgesVector
  (u/if-schema
   {(s/named s/Any "parent node")
    [[(s/one s/Any "edge label or properties")
      (s/one s/Any "child node")]]}))

(def Node
  (u/if-schema
   (s/named s/Any "node")))

(def Graph
  (u/if-schema
   {(s/required-key :edges) AdjacentEdges
    (s/required-key :nodes) #{Node}
    s/Any s/Any}))

(def GraphDiff
  (u/if-schema
   {(s/optional-key :edges) {(s/optional-key :-) (s/either AdjacentEdges
                                                           AdjacentEdgesVector)
                             (s/optional-key :+) (s/either AdjacentEdges
                                                           AdjacentEdgesVector)}
    (s/optional-key :nodes) {(s/optional-key :-) (s/either [Node] #{Node})
                             (s/optional-key :+) (s/either [Node] #{Node})}}))

;; A graph diff function takes a graph and returns a sequence of graph diffs
;; as defined by the above schema.
;;
;; It is the responsibility of the graph diff functions to only return diffs
;; that leave the graphs in a consistent and valid state.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Graph diff application and transformation functions
;;; ===================================================

(def
  ^{:dynamic true
    :doc "Bound to the current graph when transforming a graph.
         This defines the context of the `core.logic` support."}
  *current-graph* {})

(defn apply-diff
  "Applies a graph diff to `graph`. See `Graph` and `GraphDiff`."
  [graph diff]
  (u/if-schema (s/validate GraphDiff diff))
  (let [nodes (-> (into (:nodes graph #{}) (-> diff :nodes :+))
                  (set/difference (-> diff :nodes :- set)))
        edges (merge-with set/difference
                          (merge-with set/union
                                      (:edges graph {})
                                      (-> diff :edges :+ u/setify-values))
                          (-> diff :edges :- u/setify-values))]
    (assoc graph
      :nodes nodes
      :edges (or (select-keys edges nodes)
                 (:edges graph {})))))

(defn transform
  "Takes a graph and a sequence of graph diff functions.  The graph
  diff functions take a graph and return a sequence of graph diffs.
  The diffs are applied one after the other (so the last diff wins).

  Each diff function gets passed the original unmodified graph. If you
  need to work with the intermediate graphs use `transform-update`
  instead.

  Rebinds `*current-graph*` to the graph before starting the
  transformation."
  [orig-graph fs]
  (u/if-schema (s/validate Graph orig-graph))
  (binding [*current-graph* orig-graph]
    (reduce (fn [graph f]
              (reduce apply-diff graph (f orig-graph)))
            orig-graph
            fs)))

(defn transform-update
  "Takes a graph and a sequence of graph diff functions.  The graph
  diff functions take a graph and return a sequence of graph diffs.
  The diffs are applied one after the other (so the last diff wins).

  Each successive diff function gets passed the updated graph after
  applying the previous diff (normal `reduce` behavior).

  Rebinds `*current-graph*` to the updated graph after each step."
  [graph fs]
  (u/if-schema (s/validate Graph graph))
  (binding [*current-graph* graph]
    (reduce (fn [graph f]
              (set! *current-graph* graph)
              (reduce apply-diff graph (f graph)))
            graph
            fs)))
