(ns ^{:author "Sławek Gwizdowski"
      :doc "Useful row/record related functions."}
    szew.io.util
    (:require [clojure-csv.core :as csv]))


;; ## Row fixers

(defn row-adjuster
  "Creates a function that will return vectors of same length as default-row.

  With missing fields set to value from the default field.

  Why?

  Because *SV does not have to be well formed, numbers of column may vary.

  How?

    ((row-adjuster [1 2 3]) [:x])
    => [:x 2 3]
    ((row-adjuster [1 2 3]) [:1 :2 :3 :4])
    => [:1 :2 :3]
  "
  ([default-row]
   (if (empty? default-row) identity
     (let [default (vec default-row)
           the-len (count default)]
       (fn adjuster
         ([a-row]
          (-> a-row
              (into (subvec default (min the-len (count a-row))))
              (subvec 0 the-len)))
         ([] default))))))


;; ## Line Splitters

(defn fixed-width-split
  "Accept vector of slice sizes. Creates a function that will accept str and
  return vector of slices.

  Why?

  Because somebody thought fixed width data records are a good thing.

  How?

    ((fixed-width-split [4 3 4]) \"Ala ma Kota.\")
    => [\"Ala \" \"ma \" \"Kota\"]
  "
  ([fields]
   {:pre [(not (empty? fields)) (every? number? fields) (every? pos? fields)]}
   (let [steps (partition 2 1 (reductions + 0 fields))
         idx-max (last (last steps))]
     (fn splitter
       ([^String row]
        (let [len (count row)]
          (assert (<= idx-max len)
                  (format "Given row too short: is %d, expected %d" len idx-max))
          (mapv (fn [[start stop]] (subs row start stop)) steps)))
        ([] (mapv (constantly "") steps))))))


;; ## Record curators

(defn field-unpacker
  "See if field contains a str/char delimiter, if so -- tries to parse as CSV.

  Why?

  Sometimes your fields have fields. (Eyes for Days.)

  How?

    (field-unpacker \\, \"ala,ma,kota\")
    => [\"ala\", \"ma\", \"kota\"]
  "
  [delimiter ^String field]
  (if-not (and (string? field) (.contains field (str delimiter)))
    field
    (first (csv/parse-csv field :delimiter delimiter))))

(defn row-field-unpacker
  "Run field unpacker over entire row.

  Why?

  Partial it away and apply to nested CSV/TSV stuff.

  How?

    (row-field-unpacker \\, [\"xnay\" \"ala,ma,kota\" \"unpackey\")
    => [\"xnay\" [\"ala\" \"ma\" \"kota\"] \"unpackey\"]
  "
  [delimiter a-row]
  (mapv (partial field-unpacker delimiter) a-row))


;; ## Record helpers

(defn recordify
  "Takes header vector (first one in a-seq call) and tails, returns seq of maps.

  Uses zipmap to make maps with head as keys and each row as vals.

  Why?

  Before we had ->SomeRecord. Maps will always be nicer than rows.

  How?

    (recordify [[:k1 :k2] [1 2] [3 4]])
    => ({:k1 1, :k2 2}, {:k1 3 :k2 4})
    (recordify [:k1 :k2] [[1 2] [3 4]])
    => ({:k1 1, :k2 2}, {:k1 3 :k2 4})
    (let [s [['a 'b] [1 2] [3 4]]]
      (recordify (mapv keywordify (first s)) (rest s)))
    => ({:a 1, :b 2}, {:a 3, :b 4})
  "
  ([head tails]
    (lazy-seq (map (partial zipmap head) tails)))
  ([a-seq]
    (lazy-seq (map (partial zipmap (first a-seq)) (rest a-seq)))))

(defn de-recordify
  "Make single map into single vector, to use on a seq go mapv and partial.

  Maps the target map over keys vector, duh. Defaults are done by merging.

  Why?

  Because maps are so nice, but it's also nice to be able to dump them back.

  How?

    (de-recordify [:k1 :k2] {:k1 0, :k2 :z})
    => [0 :z]
    (de-recordify [:k1 :k2 :k3] {:k3 :v3} {:k1 0, :k2 :z})
    => [0 :z :v3]
    (mapv (partial de-recordify [:k1 :k2]) [{:k1 0, :k2 :z} {:k1 1, :k2 :y}])
    => [[0 :z] [1 :y]]
  "
  ([keys defaults a-map]
    (mapv (merge defaults a-map) keys))
  ([keys a-map]
    (mapv a-map keys)))

(defn keywordify
  "Take something, make it into keyword.

  Why?

  Sometimes you just need a keyword.

  Consider dedicated library: camel-snake-kebab/->kebab-case

  How?

    (keywordify \"Account: OPEX\")
    => :account-opex
  "
  [a-something]
  (-> a-something
      (str)
      (.toLowerCase)
      (.replaceAll "\\s" "_")      ; whites      -> _
      (.replaceAll "[^\\w]" "_")   ; non wordly  -> _
      (.replaceAll "_+" "_")       ; many _      -> _
      (.replace \_ \-)             ; _           -> -
      (.replaceAll "^-+" "")       ; beginning - out
      (.replaceAll "-+$" "")       ; ending - out
      (.replaceAll "-+" "-")
      (keyword)))

(defn bastardify
  "Take something, keywordify it like a SQL engine would in 1989.

  Why?

  Underscores are OK for things like H2, hyphens? Not so much.

  Consider dedicated library: camel-snake-kebab/->snake_case

  How?

    (bastardify \"Account: OPEX\")
    => :account_opex
  "
  [a-something]
  (-> a-something
      (str)
      (.toLowerCase)
      (.replaceAll "\\s" "_")      ; whites      -> _
      (.replaceAll "[^\\w]" "_")   ; non wordly  -> _
      (.replaceAll "_+" "_")       ; many _      -> _
      (.replaceAll "^_+" "")       ; beginning - out
      (.replaceAll "_+$" "")       ; ending - out
      (keyword)))

(defn friendlify
  "Take Clojure function and make it a pretty printable String.

  Why?

  Display what's running currently in a readable way.

  How?

    (friendlify friendlify)
    => \"szew.io.util/friendlify\"
  "
  [a-something]
  (-> a-something
      (class)
      (str)
      (.replaceFirst "(^class )" "")
      (.replaceFirst "(__\\d+$)" "")
      (.replaceFirst "(@[a-fA-F0-0]+$)" "")
      (.replaceAll "_BANG_" "!")
      (.replaceAll "_BAR_" "|")
      (.replaceAll "_SHARP_" "#")
      (.replaceAll "_PERCENT_" "%")
      (.replaceAll "_AMPERSTAND_" "&")
      (.replace \$ \/)
      (.replaceAll "_QMARK_" "?")
      (.replaceAll "_PLUS_" "+")
      (.replaceAll "_STAR_" "*")
      (.replaceAll "_LT_" "<")
      (.replaceAll "_GT_" ">")
      (.replaceAll "_EQ_" "=")
      (.replaceAll "_DOT_" ".")
      (.replaceAll "_COLON_" ":")
      (.replaceAll "_SINGLEQUOTE_" "'")
      (.replace \_ \-)
      ))

(defn getter
  "Return a getter function with default.

  Why?

  Builtin `get` takes map as first argument, extra bad for partial.

  How?

    (mapv (getter \"yolo\" :no) [{\"yolo\" :yes} {:wat? :wat}])
    => [:yes :no]
    (meta (getter \"yolo\" :no))
    => {:key \"yolo\", :default :no}
    (mapv (getter 1 :no) [[:yes] []])
    => [:yes :no]
    (meta (getter 1 :no))
    => {:key 1, :default :no}
  "
  ([a-key default]
    (-> (fn sub-getter [gettable]
          (get gettable a-key default))
        (with-meta {:key a-key :default default})))
  ([a-key]
    (getter a-key nil)))

(defn juxt-map
  "Give keys and values, get a juxt map of keys-values. Hint: zipmap over juxt.

  How?

    ((juxt-map :+ inc :- dec := identity) 2)
    => {:+ 3, :- 1, := 2}
  "
  [& keys-fns]
  {:pre [(-> keys-fns count even?)]}
  (let [parts  (partition 2 2 nil keys-fns)
        keysv  (mapv first parts) 
        juxt*  (apply juxt (map second parts))]
    (-> (fn juxt-mapper [& args]
          (zipmap keysv (apply juxt* args)))
        (with-meta {:keys-fns keys-fns}))))

; vim:tw=80 cc=+1 ts=2 sw=2 et ai

