# szew/h2

Clojure wrapper for H2 database.

[![szew/io](https://clojars.org/szew/h2/latest-version.svg)](https://clojars.org/szew/h2)

[API Codox][latest], [Changelog][changelog]

Breaking changes in 0.3.0

**Table of Contents**

[TOC]

---

## Why

I've been dogfooding my private Clo*j*ure toolbox (named `szew`) since 2012.

Splitting out and releasing non-proprietary parts.

## Usage

Swiftly `spec`ify H2 database connection map, URL, Data Source or Connection
Pool for `org.clojure/java.jdbc` and `seancorfield/next.jdbc`. That's mostly
it -- get a quick embedded disk based, in memory or TCP based connection going.

Now also with handy `org.h2.tools.Server` wrapper in `szew.h2.server`: allows
you to create and manage instances of PG, TCP and Web Console.

### Spec

```clojure
user=> (require '[szew.h2 :refer [spec spec->URL spec->DS spec->CP])
nil

user=> (spec "dbname")
{:classname "org.h2.Driver"
 :password ""
 :subname "nio:dbname;COMPRESS=TRUE;EARLY_FILTER=TRUE"
 :subprotocol "h2"
 :user "sa"}

user=> (spec "dbname" {:method :mem})
{:classname "org.h2.Driver"
 :password ""
 :subname "mem:dbname;DB_CLOSE_DELAY=-1;COMPRESS=TRUE;EARLY_FILTER=TRUE"
 :subprotocol "h2"
 :user "sa"}

user=> (spec "dbname" {:method :tcp})
{:classname "org.h2.Driver"
 :password ""
 :subname "tcp://localhost:9092/dbname;COMPRESS=TRUE;EARLY_FILTER=TRUE"
 :subprotocol "h2"
 :user "sa"}

```

### URL, Data Source and Connection Pool

You can also get URL, H2 Data Source and Connection pool from a spec:

```clojure
user=> (spec->URL (spec "dbname"))
"jdbc:h2:nio:dbname;COMPRESS=TRUE;EARLY_FILTER=TRUE;DEFRAG_ALWAYS=FALSE"

user=> (spec->DS (spec "dbname"))
#object[org.h2.jdbcx.JdbcDataSource 0x740970ab "ds1: url=jdbc:h2: (...)"]

user=> (spec->CP (spec "dbname"))
#object[org.h2.jdbcx.JdbcConnectionPool 0x71cffa53 "org.h2.jdbcx.(...)"]

```

### Connection Parameters

Second argument to `spec`, the `opts` map, is:

```clojure
{:method   :default
 :user     "sa"
 :password ""
 :hostname "localhost"
 :port     9092
 :split?   false
 :part     30
 :flags    {"OPTION" "VALUE"}}
```

The `:method` sets database access mode and can be one of:

```clojure
#{:default
  :nio
  :raw
  :tcp
  :mem
  :nio-mapped
  :nio-mem-fs
  :nio-mem-lzf
  :raf
  :async}
```

The `:default` method is NIO. See `defaults` and `legacy-defaults` (disable
BETA features of H2) in `szew.h2`. See doc string of `szew.h2/spec`.

### Admin Tasks

You can find some utility wrappers for H2 tasks, like `dump!`, `pump!`,
`raze!`, `shutdown!` and `copy!`, one set per library under `szew.h2.java-jdbc`
and `szew.h2.next-jdbc`.

Remember that including H2 in your application provides you with all the tools
included within that jar. Please consult H2 docs for details.

## Server Modes

H2 comes with several server options included: a PostgreSQL compatibility mode,
a TCP server and a fancy Web Console, which allows to visually inspect
currently running instances. Most methods of `org.h2.tools.Server` class were
mapped in `szew.h2.server` namespace.

Server arguments were left "as is", because these are bound to the command
line interface and might vary between H2 versions.

Reference: [`org.h2.tools.Server` javadoc][h2server].

## Bonus HOWTO: Extending H2 with Clojure

H2 can be extended with Java and we can ride on that with Java interop:

```clojure
(ns my-procedures
  (:gen-class
    :name "my_procedures"
    :main false
    :methods [
     ^{:static true}
     [long2str [java.lang.Long] java.lang.String]
     ^{:static true}
     [xnay [java.sql.Connection java.lang.Long] java.sql.ResultSet]]))


;; This will be a function
(defn -long2str [a-long]
  (str a-long))

;; This stored procedure will just run code present in query q
(defn -xnay [^java.sql.Connection conn ^String q]
  ;;java.sql.ResultSet is expected here
  (.executeQuery (.createStatement conn) q))
```

Then compile and register in H2 via SQL:

```sql
CREATE ALIAS F FOR "my_procedures.long2str";
CREATE ALIAS P FOR "my_procedures.xnay";
```

And call with SQL:

```sql
CALL F(100);
-- => "100"
CALL P('SELECT * FROM SOMETABLE');
-- => results of that select
```

## Migrating from 0.2.x to 0.3.0

Spec storage was removed in version 0.3.0, these are gone:

* `szew.h2/make!`
* `szew.h2/conn!`
* `szew.h2/dispose!`
* `szew.h2/adjust!`
* `szew.h2/connections` storage
* `szew.h2/in-memory!`
* `szew.h2/local!`

Support for `next.jdbc` was introduced, so convenience functions to export,
import, clear and copy databases got extracted into `szew.h2.java-jdbc`
and `szew.h2.next-jdbc`. These are:

* `bulk-insert-mode!`
* `retrieval-mode!`
* `dump!`
* `pump!`
* `raze!`
* `shutdown!`
* `copy!`

The misc helpers `szew.h2/de-clob` and `szew.h2/columnlist?` were moved into
`szew.h2.util` namespace.

The `:part` parameter was changed from 2GiB to 1GiB. Also it means something
special for nioMemLZF: percentage of uncompressed data blocks.

## License

Copyright © 2012 - 2020 Sławek Gwizdowski

MIT License, text can be found in the LICENSE file.

[latest]: https://spottr.bitbucket.io/szew-h2/latest/
[changelog]: CHANGELOG.md
[h2docs]: https://h2database.com/html/quickstart.html
[h2server]: http://h2database.com/javadoc/org/h2/tools/Server.html
