;; Copyright (c) Sławek Gwizdowski
;;
;; Permission is hereby granted, free of charge, to any person obtaining
;; a copy of this software and associated documentation files (the "Software"),
;; to deal in the Software without restriction, including without limitation
;; the rights to use, copy, modify, merge, publish, distribute, sublicense,
;; and/or sell copies of the Software, and to permit persons to whom the
;; Software is furnished to do so, subject to the following conditions:
;;
;; The above copyright notice and this permission notice shall be included
;; in all copies or substantial portions of the Software.
;;
;; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
;; OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
;; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
;; THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
;; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
;; FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
;; IN THE SOFTWARE.
;;
(ns ^{:author "Sławek 'smg' Gwizdowski"
      :doc "Diffing utilities of last resort. Clojurefied difflib!"}
    szew.fi.diff
    (:gen-class)
    (:import [difflib DiffUtils
                      Delta
                      Delta$TYPE
                      Chunk
                      Patch
                      DiffRowGenerator
                      DiffRowGenerator$Builder
                      DiffRow
                      DiffRow$Tag]))


;;
;; ## Patching & unidiff!
;;

(defn ->Patch
  "Diff pair of line sequences/vectors. Gives Patch back.
  "
  [original revised]
  (DiffUtils/diff (vec original) (vec revised)))

(defn unidiff
  "Return Unified Diff (unidiff) format as String.

  Arguments:

  * orig-path original file path and optional timestamp, for diff header
  * new-path revised file path and optional timestamp, for diff header
  * orig-lines original lines used as base for diff
  * a-Patch is expected to be a Patch instance
  * context-lines is a number of lines to include as context.
  "
  [orig-path rev-path orig-lines a-Patch context-lines]
  (DiffUtils/generateUnifiedDiff orig-path
                                 rev-path
                                 (vec orig-lines)
                                 a-Patch
                                 context-lines))

;;
;; ## Diffing: line by line, side by side.
;;

(defn line-maps
  "Run diff line by line and return line by line change.

  Each map will look like this:

    {:tag  #{:insert :delete :change :equal}
     :new  #{String nil}
     :old  #{String nil}
     :line long}

  What we call QUICK&DIRTY here, may be fun for quick side by side.
  "
  [original revised]
  (if (and (empty? original) (empty? revised))
    []
    (let [builder-gen  (doto (DiffRowGenerator$Builder.)
                         (.showInlineDiffs false)
                         (.columnWidth 16192))
          builder      (.build builder-gen)
          line-by-line (fn row-proc [^DiffRow row line-no]
                         (let [tag ^DiffRow$Tag (.getTag row)]
                           (condp = tag
                             DiffRow$Tag/INSERT
                             {:tag :insert
                              :new (.getNewLine row)
                              :line line-no}
                             DiffRow$Tag/DELETE
                             {:tag :delete
                              :old (.getOldLine row)
                              :line line-no}
                             DiffRow$Tag/CHANGE
                             {:tag :change
                              :old (.getOldLine row)
                              :new (.getNewLine row)
                              :line line-no}
                             DiffRow$Tag/EQUAL
                             {:tag :equal
                              :old (.getOldLine row)
                              :line line-no}
                             ; default: shout!
                             (throw (ex-info "Danger Zone: Wrong DiffRow!"
                                             {:tag :mu
                                              :row row
                                              :line line-no})))))]
      (map line-by-line (.generateDiffRows builder
                                           (vec original)
                                           (vec revised))
           (range)))))

;;
;; ## Diffing: Serious business, deltas and chunks, yo!
;;

(defn ->Deltas
  "Run Diff on pair of line sequences/vectors. Gives sequence of Deltas.
  "
  [original revised]
  (.getDeltas ^Patch (->Patch original revised)))

(defn Chunk->map
  "Creates a hash-map from Chunk instance.

  Returned map will be something like this:

    {:position long
     :span     long
     :lines    vector}
  "
  [^Chunk a-chunk]
  {:position (long (.getPosition a-chunk))
   :span     (long (.size a-chunk))
   :lines    (vec (.getLines a-chunk))})

(defn Delta->map
  "Creates a hash-map from Delta instance.

  Returned map will look something like this:

    {:tag      #{:insert :delete :change}
     :original #{chunk-map nil}
     :revised  #{chunk-map nil}}
  "
  [^Delta a-delta]
  (condp = (.getType a-delta)
    Delta$TYPE/INSERT
    {:type     :insert
     :revised  (Chunk->map (.getOriginal a-delta))}
    Delta$TYPE/DELETE
    {:type     :delete
     :original (Chunk->map (.getOriginal a-delta))}
    Delta$TYPE/CHANGE
    {:type     :change
     :original (Chunk->map (.getOriginal a-delta))
     :revised  (Chunk->map (.getRevised a-delta))}
    ; default: shout!
    (throw (ex-info "Something is wrong, man" {:delta a-delta}))))

(defn delta-maps
  "Run diff, get Deltas, publish them as hash-maps in a vector.

  LPT: consult docs of Delta->map and Chunk->map.
  "
  [original revised]
  (if (or (and (empty? original) (empty? revised)) (= original revised))
    []
    (mapv Delta->map (->Deltas original revised))))

