; Copyright (c) Sławek Gwizdowski
;
; Permission is hereby granted, free of charge, to any person obtaining
; a copy of this software and associated documentation files (the "Software"),
; to deal in the Software without restriction, including without limitation
; the rights to use, copy, modify, merge, publish, distribute, sublicense,
; and/or sell copies of the Software, and to permit persons to whom the
; Software is furnished to do so, subject to the following conditions:
;
; The above copyright notice and this permission notice shall be included
; in all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
; OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
; THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
; FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
; IN THE SOFTWARE.
;
(ns ^{:author "Sławek Gwizdowski"
      :doc "Essbase XML Outline export.

Provides szew.io/XML processors for dimension extraction. Allows both
sequencing and zipping over dimensions and members.

Just keep in mind that parsing big, deeply nested XMLs is a memory hog.
"}
 szew.essbase.otl
  (:require [clojure.zip :as zip]
            [szew.io :as io]
            [camel-snake-kebab.extras :refer [transform-keys]]
            [camel-snake-kebab.core :refer [->kebab-case-keyword]]))

;; Helpers

(defn make-good
  "Transform keys in map into kebab-case keywords.
  "
  [a-map]
  (transform-keys ->kebab-case-keyword a-map))

(defn list-content-tags
  "List all content tags visible from root of XML.
  "
  [xml]
  (->> xml :content (mapv :tag)))

;; Working with things that have :members, like parsed <Dimension> or <Member>

(defn zipper
  "Given processed dimension or member, return a zipper.

  Pro-tip: they better have :members, or this will be a short ride, aight?
  "
  [root]
  (zip/zipper (comp seq :members)
              :members
              (fn [node children] (assoc node :members children))
              root))

(defn member-seq
  "Your root into lazy-seq of members! Root first. DFS from there.

  So if you feed it a dimension -- you will get dimension first.

  If you give it a husk of {:members seq-of-dimensions} you'll get that first.
  "
  [root]
  (letfn [(walker [{:keys [members] :as head} line]
            (when head
              (cons head
                    (if (seq members)
                      (lazy-seq (walker (first members)
                                        (concat (rest members) line)))
                      (lazy-seq (walker (first line)
                                        (rest line)))))))]
    (walker root '())))

(defn member-walk
  "Given processed dimension visit and maybe modify members.

  Depth-first. Does visit root.

  That visitor callable should accept loc and return loc.
  "
  [visitor root]
  (loop [loc (zipper root)]
    (if (zip/end? loc)
      (zip/root loc)
      (recur (zip/next (visitor loc))))))

;; TODO: get meta from Element when it becomes available (alpha feature)
(defn member
  "Given parsed XML <Member> node returns processed single member.
  "
  [node]
  (let [alias-pair (juxt (comp :table :attrs) (comp first :content))
        attribute-pair (juxt (comp :dimension :attrs) (comp :name :attrs))
        content (group-by :tag (:content node))
        aliases (into (hash-map) (map alias-pair (:Alias content)))
        udas (into (hash-set) (mapcat :content (:UDA content)))
        attrs (into (hash-map) (map attribute-pair (:AttributeMember content)))
        children (count (:Member content))]
    (-> (:attrs node)
        (make-good)
        (assoc :tag :Member)
        (assoc :contained (list-content-tags node))
        (assoc :aliases aliases)
        (assoc :alias (str (first (keep aliases ["Default", "Long Names"]))))
        (assoc :uda udas)
        (assoc :attributes attrs)
        (assoc :members [])
        (assoc :level0? (zero? children))
        (update :consolidation (fnil identity "+"))
        (update :data-storage (fnil identity "StoreData"))
        (update :shared-member {"Y" true, nil false})
        (update :two-pass-calc {"Y" true, nil false}))))

(defn expanded-member
  "Given parsed XML <Member> root returns processed member with descendants.

  Expecting at to be path above root, defaults to [\"?\"].
  "
  ([at root]
   (let [pq (clojure.lang.PersistentQueue/EMPTY)
         children (comp (partial into pq
                                 (filter (comp (partial = :Member) :tag)))
                        :content)
         start (assoc (member root) :path at
                      :generation (count at)
                      :dimension (first at)
                      :parent (last at))
         push (fn [path child]
                (let [node (member child)
                      up (peek path)
                      node+ (assoc node :path (conj (:path up) (:name up))
                                   :generation (inc (:generation up))
                                   :dimension (first (:path up))
                                   :parent (:name up))]
                  (conj path node+)))]
     (loop [path [start]
            stack [(children root)]]
       (let [node (peek path)       ;; current member
             todo (peek stack)]     ;; current children to visit
         (if (seq todo) ;; work on next child
           (recur (push path (peek todo))
                  (-> (pop stack)      ;; kick todo out
                      (conj (pop todo) ;; append modified todo
                            (conj (children (peek todo)))))) ;; more children!
           (if (seq (pop path)) ;; we're not done
             (let [up (peek (pop path))
                   magnitude (pop (pop path))]
               (recur (conj magnitude (update up :members conj node))
                      (pop stack)))
             node))))))
  ([root]
   (expanded-member ["?"] root)))

;; TODO: get meta from Element when it becomes availlable (alpha feature)
(defn dimension
  "Given parsed XML <Dimension> node returns processed dimension.
  "
  [node]
  (let [gen-name (juxt #(Integer/parseInt (:number %)) :name)
        content (group-by :tag (:content node))
        gen-level (comp (partial into (sorted-map))
                        (partial map (comp gen-name :attrs))
                        (partial filter (comp (partial = :GenLevel) :tag))
                        (partial mapcat :content))
        attributes (->> (:AttributeDimension content)
                        (map (comp :name :attrs))
                        (into (hash-set)))]
    (-> (:attrs node)
        (make-good)
        (assoc :tag :Dimension)
        (assoc :contained (list-content-tags node))
        (assoc :generations (gen-level (:Generations content)))
        (assoc :levels (gen-level (:Levels content)))
        (assoc :attribute-nodes attributes)
        (assoc :members [])
        (update :dimension-type (fnil identity "Standard"))
        (update :data-storage (fnil identity "StoreData"))
        (update :density (fnil identity "Sparse"))
        (update :is-compression-dimension {"Y" true, nil false}))))

(defn expanded-dimension
  "Given parsed XML <Dimension> returns processed dimension with descendants.
  "
  [node]
  (let [root (dimension node)
        members (->> (:content node)
                     (filter (comp (partial = :Member) :tag))
                     (map (partial expanded-member [(:name root)])))
        expanded (assoc root :members members)
        member-count (dec (count (member-seq expanded)))]
    (assoc expanded :member-count member-count)))

(defn list-dimensions
  "Given parsed XML returns dimensions without members. Lazy!
  "
  [xml-root]
  (->> (:content xml-root)
       (filter (comp (partial = :Dimension) :tag))
       (map dimension)))

(defn extract-dimensions
  "Given parsed XML extracts dimensions with `expanded-dimension`. Lazy!

  Predicate `wanted?` will be fed output of `dimension`, so no `:members`!
  "
  ([xml-root]
   (->> (:content xml-root)
        (filter (comp (partial = :Dimension) :tag))
        (map expanded-dimension)))
  ([wanted? xml-root]
   (->> (:content xml-root)
        (filter (comp (partial = :Dimension) :tag))
        (filter (comp wanted? dimension))
        (map expanded-dimension))))

(defn member-set
  "Given root with :members returns set of member names, including root.
  "
  [root]
  (into (hash-set) (map :name (member-seq root))))

(defn lut-fn
  "Given list of <Dimension>s prepares a LUT function that can be called like:

  `(lut member-name)` or `(lut default member-name)`

  Attached `meta` contains:
  * :dimensions - given Dimensions (sans members, of course),
  * :lut - the LUT data structure,
  * :total-dims - total number of dimensions given,
  * :attr-dims - number of attribute dimensions given,
  * :data-dims - number of non-attribute dimensions given (used in data files).
  "
  [dimensions]
  (let [lut (->> dimensions
                 (map (juxt member-set :name))
                 (sort-by (comp count first))
                 (reverse)
                 (into []))
        total (count dimensions)
        attrs (->> dimensions
                   (filter (comp (partial = "Attribute") :dimension-type))
                   (count))
        data (- total attrs)]
    (with-meta (fn member-to-dimension-resolver
                 ([m]
                  (loop [remaining lut]
                    (let [[members dimension] (first remaining)]
                      (if (contains? members m)
                        dimension
                        (when (seq remaining)
                          (recur (rest remaining)))))))
                 ([default m]
                  (or (member-to-dimension-resolver m) default)))
      {:lut lut
       :dimensions (mapv #(assoc % :members []) dimensions)
       :total-dims total
       :attr-dims attrs
       :data-dims data})))

(defn member-lut
  "Given parsed XML returns function of member name to dimension name.

  Generated look-up-table can be found in function's meta.
  "
  ([xml-root]
   (lut-fn (extract-dimensions xml-root)))
  ([wanted? xml-root]
   (lut-fn (extract-dimensions wanted? xml-root))))

(defn data-member-lut
  "Rides on `member-lut`, but skips attribute dimensions.
  "
  [xml-root]
  (let [non-attr (comp (partial not= "Attribute") :dimension-type)]
    (member-lut non-attr xml-root)))

(defn lut-package
  "Prepares a hash-map with :m->d and :dim-count based on given file.
  "
  [xml-file]
  (let [lut (io/in! (io/xml {:processor member-lut}) xml-file)]
    {:m->d lut
     :dim-count (-> lut meta :data-dims)}))

;; Working with <cube>

(defn cube
  "Given parsed XML returns <cube> processed.
  "
  [xml-root]
  :to-be-implemented)

;; Working with <smartLists>

(defn smart-lists
  "Given parsed XML returns <smartLists> processed.
  "
  [xml-root]
  :to-be-implemented)
