; Copyright (c) Sławek Gwizdowski
;
; Permission is hereby granted, free of charge, to any person obtaining
; a copy of this software and associated documentation files (the "Software"),
; to deal in the Software without restriction, including without limitation
; the rights to use, copy, modify, merge, publish, distribute, sublicense,
; and/or sell copies of the Software, and to permit persons to whom the
; Software is furnished to do so, subject to the following conditions:
;
; The above copyright notice and this permission notice shall be included
; in all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
; OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
; THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
; FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
; IN THE SOFTWARE.
;
(ns ^{:author "Sławek Gwizdowski"
      :doc "Essbase application log and MaxL spool.

Log timestamp line looks like:

  [Tue Nov 06 08:50:26 2001]Local/Sample///Info(1013214)

So it's [timestamp]Local/application/database/issuer/type(code)

And then data follows, that looks like this:

  Clear Active on User [admin] Instance [1];

So this one contains user info, but it could be Command [.+]
or Database [.+] etc.

TODO: break entries into fields, not only headers?

Fields you'll get using AppLog:

* full timestamp, decoded from date
* date: yyyy-mm-dd decoded from timestamp
* application: String
* database: String
* user: String
* level: Info | Warning | Error | ???
* code: int
* raw: String, full payload of the entry (head + message)

Additional tables of use: Code categories [1].

And then there's MaxL allowing you to see all the useful system properties,
but only if you remember to set the column_width just right, or it will just
truncate those space padded, fixed width table outputs.

I just set it to 256 and that's the default value here. YMMV.

That MaxLSpool will help you extract those, remove the padding and pack columns
into hash maps. Will also keep MaxL output preceding and past tabular output.

Some 'special' values are resolvable via maxl-constants map. It's WIP.

[1] https://docs.oracle.com/cd/E12825_01/epm.111/esb_dbag/dlogs.htm
"}
 szew.essbase.logs
  (:gen-class)
  (:require [clojure.string :as string]
            [clj-time.format :as time.format]
            [szew.io :as io]
            [szew.io.util :refer [fixed-width-split recordify]]
            [camel-snake-kebab.core :as c-s-k]
            [clojure.java.io :as clj.io :refer [reader]])
  (:import [java.util Locale]
           [java.io BufferedReader]))

(defrecord AppLog [processor]
  io/Input
  (in! [spec source]
    (letfn [(->entry [raw-entry]
              (let [ts-in (time.format/with-locale
                            (time.format/formatter "EEE MMM dd HH:mm:ss yyyy")
                            Locale/US)
                    ts-out (time.format/with-locale
                             (time.format/formatters :date-hour-minute-second)
                             Locale/US)
                    head-re #"(?xi)^\[([^\]]+)\]  # timestamp
                                    ([^/]*)/      # source?
                                    ([^/]*)/      # app
                                    ([^/]*)/      # db
                                    ([^/]*)/      # user
                                    (?:([^/]*)/|) # unk
                                    ([^/]*)       # level
                                    \((\d+)\)$    # code"
                    head-fi (->> raw-entry
                                 first
                                 second
                                 (re-find head-re)
                                 rest
                                 (mapv str))
                    entry (-> [:mark :source :app :db :user :unk :level :code]
                              (zipmap head-fi)
                              (update :code #(Integer/parseInt %))
                              (assoc :raw (mapv second raw-entry)))
                    d-t (time.format/parse ts-in (:mark entry))
                    t-s (time.format/unparse ts-out d-t)]
                (with-meta (assoc entry :datetime d-t :timestamp t-s)
                  {:head-line (-> raw-entry first first)
                   :raw-entry raw-entry
                   :source source})))
            (bracket? [^String line]
              (.startsWith line "["))
            (entries [a-line-seq]
              (->> a-line-seq
                   (map-indexed vector)
                   (filter (comp (partial not= "") second))
                   (partition-by (comp bracket? second))
                   (partition 2 2 nil)
                   (map (comp (partial apply into)
                              (juxt (comp vec first) (comp vec second))))
                   (map ->entry)))]
      (io! "Reading file!"
           (with-open [^BufferedReader r (reader source :encoding "UTF-8")]
             (processor (entries (line-seq r))))))))

(defn app-log
  "Just zip entry headers with their payload, then feed to the processor.
  "
  ([spec]
   (into (app-log) spec))
  ([]
   (AppLog. vec)))

;; MaxL

(defn maxl-prompt?
  "Tries to find MaxL shell prompt in the beginning of given line.
  "
  [^String line]
  (re-matches #"(?xi)^(MAXL>\s+).*" line))

(defrecord MaxLSpool [processor column-width encoding]
  io/Input
  (in! [spec source]
    (letfn [(table? [row]
              (zero? (rem (count row) column-width)))
            (trimmer [row]
              (mapv #(.trim ^String %) row))
            (splitter [data]
              (let [columns (/ (count (first data)) column-width)
                    widths  (vec (repeat columns column-width))]
                (comp trimmer (fixed-width-split widths))))
            (rip [block]
              (if-not (first (filter (comp table? second) block))
                (with-meta {:head (mapv second block), :body [], :tail []}
                           {:head-line (-> block first first),
                            :source source,
                            :block block})
                (let [[head data tail] (partition-by (comp not table? second)
                                                     block)
                      slicer (splitter (map second data))
                      fields (->> data
                                  first
                                  second
                                  slicer
                                  (mapv c-s-k/->kebab-case-keyword))
                      entries (->> data
                                   (map (comp slicer second))
                                   (drop 2)
                                   (recordify fields))]
                  (with-meta {:head (trimmer (mapv second head)),
                              :data (vec entries),
                              :tail (trimmer (mapv second tail))}
                             {:head-line (-> head first first),
                              :data-line (-> data first first),
                              :tail-line (-> tail first first),
                              :source    source,
                              :block     block}))))
            (splice [lines]
              (let [prompt? (comp maxl-prompt? second)
                    blocks  (->> lines
                                 (map vector (range))
                                 (filter (comp not empty? second))
                                 (drop-while (comp not prompt?))
                                 (partition-by prompt?)
                                 (partition 2 2 nil)
                                 (map (comp (partial apply into)
                                            (juxt (comp vec first)
                                                  (comp vec second)))))]
                      (map rip blocks)))]
      (io! "Reading files here!"
           (with-open [^BufferedReader r (reader source :encoding encoding)]
             (processor (splice (line-seq r))))))))

(defn maxl-spool
  "Process MaxL spool files, packs are data from prompt to prompt.

  Gives:

  {:head [String], :data [{Keyword String}], :tail [String]}
  ;; or
  {:head [String], :data [], :tail []}

  MaxL spool setup:

  MAXL> set column_width 256;

  MAXL> set timestamp on;

  MAXL> spool on to '/your/file/location.log';

  And then process output:

  * Split by maxl-prompt? into blocks.
  * If no lines in block are multiple of column-width:
    - Head: just trimmed lines from the block
    - Data: empty vector
    - Tail: empty vector
  * If any lines in block are multiples of column-width:
    - Head: prompt (lines before wide lines)
    - Data: data (fixed-width tabular data, split, trimmed, recordified)
    - Tail: additional output at the end: INFO, WARNING, timestamp etc.

  Each returned map is having some meta attached, might want to take a peek.
  "
  ([spec]
   (into (maxl-spool) spec))
  ([]
   (MaxLSpool. vec 256 "UTF-8")))

(def ^{:doc "Some predefined values from Oracle docs."}
 maxl-constants
  {:display-application
   {:application-type {"0" (str "Unspecified encoding type. "
                                "The application was created using a "
                                "pre-Release 7." 0 " version of Essbase.")
                       "1" "This value is not in use."
                       "2" "Non-Unicode-mode application"
                       "3" "Unicode-mode application"}
    :application-status {"0" "Not Loaded"
                         "1" "Loading"
                         "2" "Loaded"
                         "3" "Unloading"}
    :storage-type {"0" "Default data storage"
                   "1" "Multidimensional data storage"
                   "2" "DB" 2 " relational data storage"}}
   :display-database
   {:db-type {"0" "Normal",
              "1" "Currency"}
    :currency-conversion {"1" "division",
                          "2" "multiplication"}
    :compression {"1" "RLE",
                  "2" "Bitmap",
                  "3" "ZLIB"}
    :pending-io-access-mode {"0" "Invalid-Error",
                             "1" "Buffered",
                             "2" "Direct"}
    :db-status {"0" "Not-Loaded",
                "1" "Loading",
                "2" "Loaded",
                "3" "Unloading"}
    :data-status {"0" "No Data",
                  "1" "Data Loaded Without Calculation",
                  "2" "Data is Calculated"}
    :request-type {"0" "Data Load",
                   "1" "Calculation",
                   "2" "Outline Update"} ;; otherwise "Unknown"
    :display-user
    {:application-access-type {"0" "No access"
                               "1" "Hyperion Essbase access"
                               "2" "Hyperion Planning access"
                               "3" "Essbase and Planning access"}
     :type {"0" "User is set up using native Essbase security."
            "1" (str "User is externally authenticated using custom "
                     "Essbase libraries.")
            "3" "User is externally authenticated using Shared Services."}}
    ;; TO BE CONTINUED
}})
