;; Copyright © technosophist
;;
;; This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of
;; the MPL was not distributed with this file, You can obtain one at http://mozilla.org/MPL/2.0/.
;;
;; This Source Code Form is "Incompatible With Secondary Licenses", as defined by the Mozilla Public
;; License, v. 2.0.
(ns systems.thoughtfull.transductio
  (:require
    [clojure.edn :as edn]
    [clojure.java.io :as io])
  (:import
    (clojure.lang IReduceInit)
    (java.io OutputStream PushbackReader Writer)
    (java.lang AutoCloseable)))

(set! *warn-on-reflection* true)

(defn decoder
  "Returns a function that takes a source and returns a reducible that, opens the source, repeatedly
  decodes data from it, and closes it when it is done.

  open takes the source and returns a stateful, mutable object that will be given to decode and
  eventually to close.

  decode is given the opened source and must return either a decoded value or a sentinel
  indicating there is no more data.

  done? is given each decoded value and if it returns truthy, then the decoding ends.

  When the reduction completes whether normally, early (someone returns a reduced), or because of
  an exception, then close is called.

  If close is not given, then it defaults to java.lang.AutoCloseable/.close.

  See stream-decoder and reader-decoder."
  ([open decode done?]
   (decoder open decode done? nil))
  ([open decode done? close]
   (fn [source]
     (let [close (or close AutoCloseable/.close)]
       (reify
         IReduceInit
         (reduce
           [_ rf init]
           (let [in (open source)]
             (try
               (loop [acc init
                      v (decode in)]
                 (if (done? v)
                   acc
                   (let [acc (rf acc v)]
                     (if (reduced? acc)
                       @acc
                       (recur acc (decode in))))))
               (finally
                 (close in))))))))))

(defn stream-decoder
  "Returns a function that takes a source and returns a reducible that, opens the source as a
  java.io.InputStream, repeatedly decodes data from it, and closes it when it is done.

  The decoder opens the source with clojure.java.io/input-stream.

  decode is given the opened source and must return either a decoded value or a sentinel
  indicating there is no more data.

  done? is given each decoded value and if it returns truthy, then the decoding ends.

  The decoder closes the source with java.lang.AutoCloseable/.close.

  See decoder, clojure-decoder, edn-decoder, and reader-decoder."
  [decode done?]
  (decoder io/input-stream decode done?))

(defn reader-decoder
  "Returns a function that takes a source and returns a reducible that, opens the source as a
  java.io.Reader, repeatedly decodes data from it, and closes it when it is done.

  The decoder opens the source with clojure.java.io/reader.  If encoding is given, then it is
  passed along to clojure.java.io/reader.

  decode is given the opened source and must return either a decoded value or a sentinel
  indicating there is no more data.

  done? is given each decoded value and if it returns truthy, then the decoding ends.

  The decoder closes the source with java.lang.AutoCloseable/.close.

  See decoder, clojure-decoder, edn-decoder, and stream-decoder."
  ([decode done?]
   (decoder io/reader decode done?))
  ([decode done? encoding]
   (decoder #(io/reader % :encoding encoding) decode done?)))

(defn- pushback-reader
  ([source encoding]
   (if (instance? PushbackReader source)
     source
     (PushbackReader. (io/reader source :encoding encoding)))))

(defn clojure-decoder
  "Returns a function that takes a source and returns a reducible that, opens the source as a
  java.io.PushbackReader, repeatedly decodes Clojure data from it, and closes it when it is done.

  The decoder opens the source as a java.io.PushbackReader.  If the source is already a
  PushbackReader, then it is used without modification.  If it is not a PushbackReader, then it is
  given to clojure.java.io/reader with the encoding (if specified) and then wrapped in a
  PushbackReader.

  If features or read-cond are specified, they are given as options to clojure.core/read.

  If data-readers, default-data-readers-fn, or read-eval are given they are bound to
  *data-readers*, *default-data-reader-fn*, and *read-eval* (respectively) for the duration of the
  reduction/transduction.

  See decoder, edn-decoder, reader-decoder, and stream-decoder."
  [& {:keys [data-readers default-data-reader-fn encoding features read-cond read-eval]}]
  (let [eof (Object.)
        opts (merge {:eof eof}
               (when (some? features)
                 {:features features})
               (when (some? read-cond)
                 {:read-cond read-cond}))
        bindings (merge {}
                   (when (some? data-readers)
                     {#'*data-readers* data-readers})
                   (when (some? default-data-reader-fn)
                     {#'*default-data-reader-fn* default-data-reader-fn})
                   (when (some? read-eval)
                     {#'*read-eval* read-eval}))
        decoder* (decoder #(pushback-reader % encoding) #(read opts %) #{eof})]
    (fn [source]
      (reify IReduceInit
        (reduce [_ rf init]
          (with-bindings bindings
            (.reduce ^IReduceInit (decoder* source) rf init)))))))

(defn edn-decoder
  "Returns a function that takes a source and returns a reducible that, opens the source as a
  java.io.PushbackReader, repeatedly decodes EDN data from it, and closes it when it is done.

  The decoder opens the source as a java.io.PushbackReader.  If the source is already a
  PushbackReader, then it is used without modification.  If it is not a PushbackReader, then it is
  given to clojure.java.io/reader with the encoding (if specified) and then wrapped in a
  PushbackReader.

  If default or readers are specified, they are given as options to clojure.edn/read.

  See decoder, clojure-decoder, reader-decoder, and stream-decoder."
  [& {:keys [default encoding readers]}]
  (let [eof (Object.)
        opts (merge {:eof eof}
               (when (some? default)
                 {:default default})
               (when (some? readers)
                 {:readers readers}))]
    (decoder #(pushback-reader % encoding)
      #(edn/read opts %)
      #{eof})))

(defn encoder
  "Returns a function that takes either a sink and a source, reducing the source into the sink, or a
  sink, a transducer, and a source, transducing the source into the sink through the transducer.
  This is an analog of into but for a mutable sink.

  open takes the sink and returns a stateful, mutable object that will be given to encode! and
  eventually to close.

  encode! is given the opened sink and a value and must mutate the sink with the value.  The
  return value from encode! is ignored.

  When the reduction/transduction completes whether normally, early (someone returns a reduced),
  or because of an exception, then close is called.

  If close is not given, then it defaults to java.lang.AutoCloseable/.close.

  See clojure-encoder, stream-encoder, and writer-encoder"
  ([open encode!]
   (encoder open encode! nil))
  ([open encode! close]
   (fn
     ([sink source]
      (let [out (open sink)
            close (or close AutoCloseable/.close)]
        (try
          (reduce (fn [out value] (encode! out value) out) out source)
          nil
          (finally
            (close out)))))
     ([sink xf source]
      (let [out (open sink)
            close (or close AutoCloseable/.close)]
        (try
          (transduce xf (fn ([out] out) ([out value] (encode! out value) out)) out source)
          nil
          (finally
            (close out))))))))

(defn stream-encoder
  "Returns a function that takes either a sink and a source, reducing the source into the sink, or a
  sink, a transducer, and a source, transducing the source into the sink through the transducer.
  This is an analog of into but for a java.io.OutputStream.

  The encoder opens the sink with clojure.java.io/output-stream.  If append is given, then it is
  passed along to clojure.java.io/output-stream.

  The source (or the transducer applied to the source) must return byte arrays which are written
  to the output stream.

  When the reduction/transduction completes whether normally, early (someone returns a reduced),
  or because of an exception, then the stream is closed.

  See encoder, clojure-encoder, and writer-encoder"
  ([]
   (encoder io/output-stream ^[bytes] OutputStream/.write))
  ([append]
   (encoder #(io/output-stream % :append append) ^[bytes] OutputStream/.write)))

(defn writer-encoder
  "Returns a function that takes either a sink and a source, reducing the source into the sink, or a
  sink, a transducer, and a source, transducing the source into the sink through the transducer.
  This is an analog of into but for a java.io.Writer.

  The encoder opens the sink with clojure.java.io/writer.  If append is given, then it is passed
  along to clojure.java.io/writer.

  The source (or the transducer applied to the source) must return Strings which are written to
  the output stream.

  When the reduction/transduction completes whether normally, early (someone returns a reduced),
  or because of an exception, then the stream is closed.

  See encoder, clojure-encoder, and stream-encoder"
  ([]
   (encoder io/writer ^[String] Writer/.write))
  ([append]
   (encoder #(io/writer % :append append) ^[String] Writer/.write)))

(defn clojure-encoder
  "Returns a function that takes either a sink and a source, reducing the source as Clojure data
  into the sink, or a sink, a transducer, and a source, transducing the source as Clojure data
  into the sink through the transducer.  This is an analog of into but for Clojure data into a
  java.io.Writer.

  The encoder opens the sink with clojure.java.io/writer.  If encoding or append are given, then
  they are passed along to clojure.java.io/writer.

  The data from the source (or the transducer applied to the source) is written to the sink using
  prn-str.

  When the reduction/transduction completes whether normally, early (someone returns a reduced),
  or because of an exception, then the stream is closed.

  See encoder, stream-encoder, and writer-encoder"
  [& {:keys [encoding append]}]
  (encoder #(io/writer % :encoding encoding :append append)
    (fn [^Writer sink value]
      (.write sink (prn-str value)))))
