(ns systems.thoughtfull.amalgam.jdbc
  (:require
    [com.stuartsierra.component :as component]
    [systems.thoughtfull.desiderata :as desiderata])
  (:import
    (java.lang AutoCloseable)
    (java.io Closeable)
    (javax.sql DataSource)
    (java.time Duration)))

(set! *warn-on-reflection* true)

(defn- data-source!
  ^DataSource [{:keys [data-source]}]
  (when-not data-source
    (throw (IllegalStateException. "DataSourceComponent is not started")))
  data-source)

(desiderata/defrecord DataSourceComponent
  "A SQL data source that is also a component.  It can be injected as a dependency of other
  components but also implements the DataSource interface.

  If the wrapped data source implements *java.io.AutoCloseable* or *java.io.Closeable*, then when
  the component is stopped it will close the data source.  If the wrapped data source does not
  implement either interface, then stop will not attempt to close it.

  - **`make-data-source-fn`** — a function that takes the DataSourceComponent as an argument and
    creates a DataSource for wrapping.  Any options necessary for constructing a DataSource should
    be taken from the DataSourceComponent.
  - **`login-timeout-duration`** (optional) — the maximum *java.time.Duration* to wait while
    attempting to connect to a database.  A value of zero specifies that the timeout is the default
    system timeout if there is one; otherwise, it specifies that there is no timeout. When a
    DataSource object is created, the login timeout is initially zero.
  - **`log-writer`** (optional) — a *java.io.PrintWriter* to set as the log writer for this data
    source, defaults to nil which disables logging."
  [make-data-source-fn login-timeout-duration log-writer]
  ::desiderata/defaults
  {:login-timeout-duration (Duration/ofSeconds 0)
   :log-writer nil}
  DataSource
  (getConnection
    [this]
    (.getConnection (data-source! this)))
  (getConnection
    [this username password]
    (.getConnection (data-source! this) username password))
  (getLoginTimeout
    [this]
    (.getLoginTimeout (data-source! this)))
  (getLogWriter
    [this]
    (.getLogWriter (data-source! this)))
  (setLoginTimeout
    [this seconds]
    (.setLoginTimeout (data-source! this) seconds))
  (setLogWriter
    [this out]
    (.setLogWriter (data-source! this) out))
  (getParentLogger
    [this]
    (.getParentLogger (data-source! this)))
  (isWrapperFor
    [this iface]
    (.isWrapperFor (data-source! this) iface))
  (unwrap
    [this iface]
    (.unwrap (data-source! this) iface))
  Closeable
  (close [this]
    (Closeable/.close (data-source! this)))
  component/Lifecycle
  (start
    [this]
    (if (:data-source this)
      this
      (assoc this :data-source (doto ^DataSource (make-data-source-fn this)
                                 (.setLoginTimeout (Duration/.toSeconds login-timeout-duration))
                                 (.setLogWriter log-writer)))))
  (stop
    [this]
    (when-let [data-source (:data-source this)]
      (when (instance? AutoCloseable data-source)
        (AutoCloseable/.close data-source)))
    (dissoc this :data-source)))

(defn ^:deprecated ^:no-doc data-source
  "Make a new DataSourceComponent wrapping a javax.sql.DataSource.

  If the wrapped data source implements *java.io.AutoCloseable* or *java.io.Closeable*, then when
  the component is stopped it will close the data source.  If the wrapped data source does not
  implement either interface, then stop will not attempt to close it.

  - **`make-data-source-fn`** — a function that takes the DataSourceComponent as an argument and
    creates a DataSource for wrapping.  Any options necessary for constructing a DataSource should
    be taken from the DataSourceComponent.
  - **`login-timeout-duration`** — the maximum *java.time.Duration* to wait while attempting to
    connect to a database.  A value of zero specifies that the timeout is the default system timeout
    if there is one; otherwise, it specifies that there is no timeout. When a DataSource object is
    created, the login timeout is initially zero.
  - **`log-writer`** — a *java.io.PrintWriter* to set as the log writer for this data source,
    defaults to nil which disables logging.

  Any additional options are passed along to the DataSourceComponent and then to
  `make-data-source-fn` when the component is started.

  See *javax.sql.DataSource*"
  {:arglists '([make-data-source-fn & {:keys [login-timeout-duration log-writer]}])}
  ^DataSource [make-data-source-fn & {:as opts}]
  (map->DataSourceComponent :make-data-source-fn make-data-source-fn opts))
