;; Copyright © technosophist
;;
;; This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of
;; the MPL was not distributed with this file, You can obtain one at http://mozilla.org/MPL/2.0/.
;;
;; This Source Code Form is "Incompatible With Secondary Licenses", as defined by the Mozilla Public
;; License, v. 2.0.
(ns systems.thoughtfull.amalgam.executors
  (:require
    [com.stuartsierra.component :as component]
    [systems.thoughtfull.desiderata :as desiderata])
  (:import
    (com.stuartsierra.component Lifecycle)
    (java.time Duration)
    (java.util.concurrent ExecutorService Executors Future LinkedBlockingQueue
      RejectedExecutionHandler ScheduledExecutorService ScheduledFuture ScheduledThreadPoolExecutor
      ThreadPoolExecutor ThreadPoolExecutor$AbortPolicy ThreadPoolExecutor$CallerRunsPolicy
      ThreadPoolExecutor$DiscardOldestPolicy ThreadPoolExecutor$DiscardPolicy TimeUnit)))

(set! *warn-on-reflection* true)

(defn conveying-runnable
  "Wrap task in a *Runnable* that conveys bindings then runs it."
  ^Runnable [^Runnable task]
  (let [frame (clojure.lang.Var/cloneThreadBindingFrame)]
    (fn []
      (clojure.lang.Var/resetThreadBindingFrame frame)
      (.run task))))

(defn conveying-callable
  "Wrap task in a *Callable* that conveys bindings then calls it."
  ^Callable [^Callable task]
  (let [frame (clojure.lang.Var/cloneThreadBindingFrame)]
    (fn []
      (clojure.lang.Var/resetThreadBindingFrame frame)
      (.call task))))

(defn all-conveying-callables
  "Wrap each task in a *Callable* that conveys bindings then calls it."
  [tasks]
  (into [] (map conveying-callable) tasks))

(defn ^:deprecated ^:no-doc executor-service
  "Return wrapped ExecutorService from ExecutorServiceComponent.  Throws IllegalStateException if
  the component is not started (i.e. it does not have an ExecutorService)."
  ^ExecutorService [{:keys [executor-service]}]
  (when-not executor-service
    (throw (IllegalStateException. "ExecutorServiceComponent is not started")))
  executor-service)

(desiderata/defrecord ^:deprecated ^:no-doc ExecutorServiceComponent
  "An executor service that is also a component.  It can be injected as a dependency of other
  components but also implements the ExecutorService interface.

  - **`make-executor-service-fn`** — a function that takes the ExecutorServiceComponent as an
    argument and creates an ExecutorService for wrapping.  Any options necessary for constructing an
    ExecutorService should be taken from the ExecutorServiceComponent.
  - **`convey-bindings?`** — if true then convey thread local var bindings with tasks as they are
    submitted to the wrapped ExecutorService, defaults to true.
  - **`termination-wait-duration`** — a *java.time.Duration* to wait for the ExecutorService to
    terminate when this component is stopped.  A duration of zero means don't wait at all.  If not
    specified, then wait indefinitely."
  [make-executor-service-fn convey-bindings? termination-wait-duration]
  ::desiderata/defaults {:convey-bindings? true}
  ExecutorService
  (awaitTermination
    [this timeout unit]
    (.awaitTermination (executor-service this) timeout unit))
  (invokeAll
    [this tasks]
    (.invokeAll (executor-service this) (cond-> tasks convey-bindings? all-conveying-callables)))
  (invokeAll
    [this tasks timeout unit]
    (.invokeAll (executor-service this) (cond-> tasks convey-bindings? all-conveying-callables)
      timeout unit))
  (invokeAny
    [this tasks]
    (.invokeAny (executor-service this) (cond-> tasks convey-bindings? all-conveying-callables)))
  (invokeAny
    [this tasks timeout unit]
    (.invokeAny (executor-service this) (cond-> tasks convey-bindings? all-conveying-callables)
      timeout unit))
  (isShutdown
    [this]
    (.isShutdown (executor-service this)))
  (isTerminated
    [this]
    (.isTerminated (executor-service this)))
  (shutdown
    [this]
    (.shutdown (executor-service this)))
  (shutdownNow
    [this]
    (.shutdownNow (executor-service this)))
  (^Future submit
   [this ^Runnable task]
   (.submit (executor-service this) (cond-> task convey-bindings? conveying-runnable)))
  (^Future submit
   [this ^Runnable task result]
   (.submit (executor-service this) (cond-> task convey-bindings? conveying-runnable) result))
  (^Future submit
   [this ^Callable task]
   (.submit (executor-service this) (cond-> task convey-bindings? conveying-callable)))
  (execute
    [this command]
    (.execute (executor-service this) (cond-> command convey-bindings? conveying-runnable)))
  component/Lifecycle
  (start
    [this]
    (if (:executor-service this)
      this
      (let [this (assoc this :convey-bindings? (boolean convey-bindings?))]
        (assoc this :executor-service (make-executor-service-fn this)))))
  (stop
    [this]
    (when-let [executor-service ^ExecutorService (:executor-service this)]
      (.shutdown executor-service)
      (if-let [termination-wait-ns (some-> termination-wait-duration Duration/.toNanos)]
        (when (pos? termination-wait-ns)
          (.awaitTermination executor-service termination-wait-ns TimeUnit/NANOSECONDS))
        (while (not (.isTerminated executor-service)))))
    (dissoc this :executor-service)))

(defn- adapt-rejected-execution-handler
  ^RejectedExecutionHandler [rejected-execution-handler]
  (case (or rejected-execution-handler :abort)
    (:abort :abort-policy) (ThreadPoolExecutor$AbortPolicy.)
    (:caller-runs :caller-runs-policy) (ThreadPoolExecutor$CallerRunsPolicy.)
    (:discard-oldest :discard-oldest-policy) (ThreadPoolExecutor$DiscardOldestPolicy.)
    (:discard :discard-policy) (ThreadPoolExecutor$DiscardPolicy.)
    (if (fn? rejected-execution-handler)
      (reify
        RejectedExecutionHandler
        (rejectedExecution
          [_this runnable executor]
          (rejected-execution-handler runnable executor)))
      rejected-execution-handler)))

(defn- caller-runs?
  [reh]
  (instance? ThreadPoolExecutor$CallerRunsPolicy reh))

(defn ^:deprecated ^:no-doc make-thread-pool
  "Make a ThreadPoolExecutor for wrapping by ExecutorServiceComponent.

  - **`core-pool-size`** — number of threads to keep in the pool, even if they are idle
  - **`max-pool-size`** — maximum number of threads to allow in the pool
  - **`keep-alive-duration`** — when the number of threads is greater than the core, the maximum
    *java.time.Duration* to wait for new tasks before terminating
  - **`work-queue`** — queue to use for holding tasks before they are executed
  - **`thread-factory`** — factory to use when executor creates a new thread.  Defaults to
    *Executors/defaultThreadFactory*.
  - **`rejected-execution-handler`** — handler to use when execution is blocked because the thread
    and queue capacities are reached.  The handler can be an instance of
    *java.util.concurrent.RejectedExecutionHandler*, a function of two arguments (a *Runnable* and a
    *ThreadPoolExecutor*), or one of `:abort`, `:caller-runs`, `:discard-oldest`, `:discard`,
    defaults to `:abort`.

  See *java.util.concurrent.ThreadPoolExecutor*"
  [{:keys [core-pool-size max-pool-size keep-alive-duration work-queue thread-factory
           rejected-execution-handler convey-bindings?]}]
  (when (and convey-bindings?
          (or (= :caller-runs rejected-execution-handler)
            (= :caller-runs-policy rejected-execution-handler)
            (caller-runs? rejected-execution-handler)))
    (throw (IllegalStateException. "convey-bindings? is not compatible with caller runs policy")))
  (let [keep-alive-ns (-> keep-alive-duration Duration/.toNanos)]
    (ThreadPoolExecutor. core-pool-size max-pool-size keep-alive-ns TimeUnit/NANOSECONDS
      work-queue (or thread-factory (Executors/defaultThreadFactory))
      (-> rejected-execution-handler adapt-rejected-execution-handler))))

(defn ^:deprecated ^:no-doc thread-pool
  "Make a new ExecutorServiceComponent wrapping a ThreadPoolExecutor.

  - **`core-pool-size`** — number of threads to keep in the pool, even if they are idle
  - **`max-pool-size`** — maximum number of threads to allow in the pool
  - **`keep-alive-duration`** — when the number of threads is greater than the core, the maximum
    *java.time.Duration* to wait for new tasks before terminating
  - **`work-queue`** — queue to use for holding tasks before they are executed
  - **`thread-factory`** — factory to use when executor creates a new thread.
  - **`convey-bindings?`** — if true then convey thread local var bindings with tasks as they are
    submitted to the wrapped ExecutorService.
  - **`rejected-execution-handler`** — handler to use when execution is blocked because the thread
    and queue capacities are reached.  The handler can be an instance of
    *java.util.concurrent.RejectedExecutionHandler*, a function of two arguments (a *Runnable* and a
    *ThreadPoolExecutor*), or one of `:abort`, `:caller-runs`, `:discard-oldest`, `:discard`,
    defaults to `:abort`.
  - **`termination-wait-duration`** — a *java.time.Duration* to wait for the ExecutorService to
    terminate when the component is stopped.  A duration of zero means don't wait at all.  If not
    specified, then wait indefinitely."
  {:arglists '([core-pool-size max-pool-size keep-alive-duration work-queue
                & {:as opts :keys [thread-factory rejected-execution-handler convey-bindings?
                                   termination-wait-duration]}])}
  ^ExecutorService [core-pool-size max-pool-size keep-alive-duration work-queue & {:as opts}]
  (map->ExecutorServiceComponent
    :core-pool-size core-pool-size
    :max-pool-size max-pool-size
    :keep-alive-duration keep-alive-duration
    :work-queue work-queue
    :make-executor-service-fn make-thread-pool
    opts))

(defn ^:deprecated ^:no-doc scheduled-executor-service
  "Return wrapped ScheduledExecutorService from ScheduledExecutorServiceComponent.  Throws
  IllegalStateException if the component is not started (i.e. it does not have an
  ScheduledExecutorService)."
  ^ScheduledExecutorService [{:keys [executor-service]}]
  (when-not executor-service
    (throw (IllegalStateException. "ScheduledExecutorServiceComponent is not started")))
  executor-service)

(desiderata/defrecord ^:deprecated ^:no-doc ScheduledExecutorServiceComponent
  "An scheduled executor service that is also a component.  It can be injected as a dependency of
  other components but also implements the ScheduledExecutorService interface.

  - **`make-executor-service-fn`** — a function that takes the ScheduledExecutorServiceComponent as
    an argument and creates a ScheduledExecutorService for wrapping.  Any options necessary for
    constructing an ScheduledExecutorService should be taken from the
    ScheduledExecutorServiceComponent.
  - **`convey-bindings?`** — if true then convey thread local var bindings with tasks as they are
    submitted to the wrapped ExecutorService.
  - **`termination-wait-duration`** — a *java.time.Duration* to wait for the
    ScheduledExecutorService to terminate when this component is stopped.  A duration of zero means
    don't wait at all.  If not specified, then wait indefinitely."
  [make-executor-service-fn convey-bindings? termination-wait-duration]
  ::desiderata/defaults {:convey-bindings? true}
  ScheduledExecutorService
  (awaitTermination
    [this timeout unit]
    (.awaitTermination (scheduled-executor-service this) timeout unit))
  (invokeAll
    [this tasks]
    (.invokeAll (scheduled-executor-service this)
      (cond-> tasks convey-bindings? all-conveying-callables)))
  (invokeAll
    [this tasks timeout unit]
    (.invokeAll (scheduled-executor-service this)
      (cond-> tasks convey-bindings? all-conveying-callables) timeout unit))
  (invokeAny
    [this tasks]
    (.invokeAny (scheduled-executor-service this)
      (cond-> tasks convey-bindings? all-conveying-callables)))
  (invokeAny
    [this tasks timeout unit]
    (.invokeAny (scheduled-executor-service this)
      (cond-> tasks convey-bindings? all-conveying-callables) timeout unit))
  (isShutdown
    [this]
    (.isShutdown (scheduled-executor-service this)))
  (isTerminated
    [this]
    (.isTerminated (scheduled-executor-service this)))
  (shutdown
    [this]
    (.shutdown (scheduled-executor-service this)))
  (shutdownNow
    [this]
    (.shutdownNow (scheduled-executor-service this)))
  (^Future submit
   [this ^Runnable task]
   (.submit (scheduled-executor-service this) (cond-> task convey-bindings? conveying-runnable)))
  (^Future submit
   [this ^Runnable task result]
   (.submit (scheduled-executor-service this) (cond-> task convey-bindings? conveying-runnable)
     result))
  (^Future submit
   [this ^Callable task]
   (.submit (scheduled-executor-service this) (cond-> task convey-bindings? conveying-callable)))
  (execute
    [this command]
    (.execute (scheduled-executor-service this)
      (cond-> command convey-bindings? conveying-runnable)))
  (^ScheduledFuture schedule
   [this ^Runnable command ^long delay ^TimeUnit unit]
   (.schedule (scheduled-executor-service this) (cond-> command convey-bindings? conveying-runnable)
     delay unit))
  (^ScheduledFuture schedule
   [this ^Callable command ^long delay ^TimeUnit unit]
   (.schedule (scheduled-executor-service this) (cond-> command convey-bindings? conveying-callable)
     delay unit))
  (scheduleAtFixedRate
    [this command initial-delay period unit]
    (.scheduleAtFixedRate (scheduled-executor-service this)
      (cond-> command convey-bindings? conveying-runnable) initial-delay period unit))
  (scheduleWithFixedDelay
    [this command initial-delay delay unit]
    (.scheduleAtFixedRate (scheduled-executor-service this)
      (cond-> command convey-bindings? conveying-runnable) initial-delay delay unit))
  component/Lifecycle
  (start
    [this]
    (if (:executor-service this)
      this
      (let [this (assoc this :convey-bindings? (boolean convey-bindings?))]
        (assoc this :executor-service (make-executor-service-fn this)))))
  (stop
    [this]
    (when-let [executor-service ^ExecutorService (:executor-service this)]
      (.shutdown executor-service)
      (if-let [termination-wait-ns (some-> termination-wait-duration Duration/.toNanos)]
        (when (pos? termination-wait-ns)
          (.awaitTermination executor-service termination-wait-ns TimeUnit/NANOSECONDS))
        (while (not (.isTerminated executor-service)))))
    (dissoc this :executor-service)))

(defn ^:deprecated ^:no-doc make-scheduled-thread-pool
  "Make a ScheduledThreadPoolExecutor for wrapping by ScheduledExecutorServiceComponent.

  - **`core-pool-size`** — number of threads to keep in the pool, even if they are idle
  - **`remove-on-cancel?`** — immediately remove canceled tasks from the queue, or wait until their
    delay elapses, defaults to false, which means wait until their delay elapses.
  - **`continue-existing-periodic-tasks-after-shutdown?`** — continue executing existing periodic
    tasks even when this executor has been shutdown, defaults to false, which means immediately
    stop them.
  - **`execute-existing-delayed-tasks-after-shutdown?`** — execute existing delayed tasks even when
    this executor has been shutdown, defaults to true, which means execute them (false means
    immediately stop them).
  - **`thread-factory`** — factory to use when executor creates a new thread.  Defaults to
    *Executors/defaultThreadFactory*.
  - **`rejected-execution-handler`** — handler to use when execution is blocked because the thread
    and queue capacities are reached.  The handler can be an instance of
    *java.util.concurrent.RejectedExecutionHandler*, a function of two arguments (a *Runnable* and a
    *ThreadPoolExecutor*), or one of `:abort`, `:caller-runs`, `:discard-oldest`, `:discard`,
    defaults to `:abort`.

  See *java.util.concurrent.ScheduledThreadPoolExecutor*"
  [{:keys [core-pool-size remove-on-cancel? continue-existing-periodic-tasks-after-shutdown?
           execute-existing-delayed-tasks-after-shutdown? thread-factory rejected-execution-handler
           convey-bindings? termination-wait-duration]}]
  (when (and convey-bindings?
          (or (= :caller-runs rejected-execution-handler)
            (= :caller-runs-policy rejected-execution-handler)
            (caller-runs? rejected-execution-handler)))
    (throw (IllegalStateException.
             "convey-bindings? is not compatible with caller runs policy, an error will occur")))
  (when (and continue-existing-periodic-tasks-after-shutdown? (nil? termination-wait-duration))
    (throw (IllegalStateException.
             (str "continue-existing-periodic-tasks-after-shutdown? is not compatible with nil"
               " termination-wait-duration, a deadlock will occur"))))
  (doto (ScheduledThreadPoolExecutor. core-pool-size
          (or thread-factory (Executors/defaultThreadFactory))
          (-> rejected-execution-handler adapt-rejected-execution-handler))
    (cond-> remove-on-cancel?
      (.setRemoveOnCancelPolicy true))
    (cond-> continue-existing-periodic-tasks-after-shutdown?
      (.setContinueExistingPeriodicTasksAfterShutdownPolicy true))
    (cond-> (false? execute-existing-delayed-tasks-after-shutdown?)
      (.setExecuteExistingDelayedTasksAfterShutdownPolicy false))))

(defn ^:deprecated ^:no-doc scheduled-thread-pool
  "Make a new ScheduledExecutorServiceComponent wrapping a ScheduledThreadPoolExecutor.

  - **`core-pool-size`** — number of threads to keep in the pool, even if they are idle
  - **`remove-on-cancel?`** — whether to immediately remove canceled tasks from the queue, or wait
    until their delay elapses, defaults to false, which means do not immediately remove them.
  - **`continue-existing-periodic-tasks-after-shutdown?`** — continue executing existing periodic
    tasks even when this executor has been shutdown, defaults to false, which means immediately
    stop them.
  - **`execute-existing-delayed-tasks-after-shutdown?`** — execute existing delayed tasks even when
    this executor has been shutdown, defaults to true, which means execute them (false means
    immediately stop them).
  - **`thread-factory`** — factory to use when executor creates a new thread.  Defaults to
    *Executors/defaultThreadFactory*.
  - **`convey-bindings?`** — if true then convey thread local var bindings with tasks as they are
    submitted to the wrapped ScheduledExecutorService.
  - **`rejected-execution-handler`** — handler to use when execution is blocked because the thread
    and queue capacities are reached.  The handler can be an instance of
    *java.util.concurrent.RejectedExecutionHandler*, a function of two arguments (a *Runnable* and a
    *ThreadPoolExecutor*), or one of `:abort`, `:caller-runs`, `:discard-oldest`, `:discard`,
    defaults to `:abort`.
  - **`termination-wait-duration`** — a *java.time.Duration* to wait for the
    ScheduledExecutorService to terminate when the component is stopped.  A duration of zero means
    don't wait at all.  If not specified, then wait indefinitely."
  {:arglists
   '([core-pool-size
      & {:as opts
         :keys [remove-on-cancel? continue-existing-periodic-tasks-after-shutdown?
                execute-existing-delayed-tasks-after-shutdown? thread-factory
                rejected-execution-handler convey-bindings? termination-wait-duration]}])}
  ^ScheduledExecutorService [core-pool-size & {:as opts}]
  (map->ScheduledExecutorServiceComponent
    :core-pool-size core-pool-size
    :make-executor-service-fn make-scheduled-thread-pool
    opts))

(defn scheduled-future
  "Return wrapped *ScheduledFuture* from *ScheduledTaskComponent*.  Throws *IllegalStateException*
  if the component is not started (i.e. it does not have an *ScheduledFuture*)."
  ^ScheduledFuture [{:keys [scheduled-future]}]
  (when-not scheduled-future
    (throw (Exception. "ScheduledTaskComponent is not started")))
  scheduled-future)

(declare map->ScheduledThreadPoolComponent)

(desiderata/defrecord ^:deprecated ^:no-doc ScheduledTaskComponent
  "A scheduled task that is also a component.

  - **`executor-service`** — ScheduledExecutorService to use to schedule task.  If not specified
    then a single thread `scheduled-thread-pool` component is started when this component is started
    and stopped when this component is stopped.
  - **`make-scheduled-future-fn`** — a function that takes the ScheduledTaskComponent as an argument
    and creates a ScheduledFuture for wrapping.  Any options necessary for constructing a
    ScheduledFuture should be taken from the ScheduledTaskComponent."
  [^ScheduledExecutorService scheduled-executor-service make-scheduled-future-fn]
  ScheduledFuture
  (compareTo
    [this other]
    (.compareTo (scheduled-future this) other))
  (getDelay
    [this unit]
    (.getDelay (scheduled-future this) unit))
  (cancel
    [this may-interrupt-if-running?]
    (.cancel (scheduled-future this) may-interrupt-if-running?))
  (get
    [this]
    (.get (scheduled-future this)))
  (get
    [this timeout unit]
    (.get (scheduled-future this) timeout unit))
  (isCancelled
    [this]
    (.isCancelled (scheduled-future this)))
  (isDone
    [this]
    (.isDone (scheduled-future this)))
  component/Lifecycle
  (start
    [this]
    (if (:scheduled-future this)
      this
      (let [my-scheduled-executor-service (when (nil? scheduled-executor-service)
                                            (component/start (map->ScheduledThreadPoolComponent)))
            scheduled-executor-service (or scheduled-executor-service my-scheduled-executor-service)
            this (assoc this
                   :scheduled-executor-service scheduled-executor-service
                   :my-scheduled-executor-service my-scheduled-executor-service)]
        (assoc this :scheduled-future (make-scheduled-future-fn scheduled-executor-service this)))))
  (stop
    [{:as this
      :keys [^ScheduledFuture scheduled-future my-scheduled-executor-service]}]
    (when scheduled-future
      (.cancel scheduled-future false))
    (when my-scheduled-executor-service
      (component/stop my-scheduled-executor-service))
    (cond-> (dissoc this :scheduled-future :my-scheduled-executor-service)
      my-scheduled-executor-service (dissoc this :scheduled-executor-service))))

(defn ^:deprecated ^:no-doc make-fixed-rate-scheduled-task
  "Make a ScheduledFuture for wrapping by ScheduledTaskComponent.

  - **`executor-service`** — ScheduledExecutorService to use to schedule task.  If not specified
    then a single thread `scheduled-thread-pool` component is started when this component is started
    and stopped when this component is stopped.
  - **`task`** — a one argument function to run as a scheduled task.  It takes as an argument the
    task componentand its dependencies.
  - **`initial-delay-duration`** — *java.time.Duration* to wait before the first run of `task`
  - **`period-duration`** — *java.time.Duration* from the start of one run to the start of the next

  See *java.util.concurrent.ScheduledThreadPoolExecutor*"
  [^ScheduledExecutorService executor-service
   {:as component :keys [task initial-delay-duration period-duration]}]
  (let [initial-delay-ns (or (some-> initial-delay-duration Duration/.toNanos) 0)
        period-ns (some-> period-duration Duration/.toNanos)]
    (.scheduleAtFixedRate executor-service #(task component) initial-delay-ns period-ns
      TimeUnit/NANOSECONDS)))

(defn ^:deprecated ^:no-doc fixed-rate-scheduled-task
  "Make a new fixed rate ScheduledTaskComponent wrapping a ScheduledFuture.

  - **`task`** — a one argument function to run as a scheduled task.  It takes as an argument the
    task component and its dependencies.
  - **`period-duration`** — *java.time.Duration* from the start of one run to the start of the next
  - **`initial-delay-duration`** — *java.time.Duration* to wait before the first run of `task`,
    defaults to 0
  - **`executor-service`** — ScheduledExecutorService to use to schedule task.  If not specified
    then a single thread `scheduled-thread-pool` component is started when this component is started
    and stopped when this component is stopped.

  See *java.util.concurrent.ScheduledThreadPoolExecutor*"
  {:arglists '([task period-duration & {:as opts :keys [initial-delay-duration executor-service]}])}
  ^ScheduledFuture [task period-duration & {:as opts}]
  (map->ScheduledTaskComponent
    :task task
    :period-duration period-duration
    :make-scheduled-future-fn make-fixed-rate-scheduled-task
    opts))

(defn ^:deprecated ^:no-doc make-fixed-delay-scheduled-task
  "Make a ScheduledFuture for wrapping by ScheduledTaskComponent.

  - **`executor-service`** — ScheduledExecutorService to use to schedule task.  If not specified
    then a single thread `scheduled-thread-pool` component is started when this component is started
    and stopped when this component is stopped.
  - **`task`** — a one argument function to run as a scheduled task.  It takes as an argument the
    task component and its dependencies.
  - **`initial-delay-duration`** — *java.time.Duration* to wait before the first run of `task`
  - **`delay-duration`** — *java.time.Duration* from the end of one run to the start of the next

  See *java.util.concurrent.ScheduledThreadPoolExecutor*"
  [^ScheduledExecutorService executor-service
   {:as component :keys [task initial-delay-duration delay-duration]}]
  (let [initial-delay-ns (or (some-> initial-delay-duration Duration/.toNanos) 0)
        delay-ns (some-> delay-duration Duration/.toNanos)]
    (.scheduleWithFixedDelay executor-service #(task component) initial-delay-ns delay-ns
      TimeUnit/NANOSECONDS)))

(defn ^:deprecated ^:no-doc fixed-delay-scheduled-task
  "Make a new fixed delay ScheduledTaskComponent wrapping a ScheduledFuture.

  - **`task`** — a one argument function to run as a scheduled task.  It takes as an argument the
    task component and its dependencies.
  - **`delay-duration`** — *java.time.Duration* from the end of one run to the start of the next
  - **`initial-delay-duration`** — *java.time.Duration* to wait before the first run of `task`,
    defaults to 0
  - **`executor-service`** — ScheduledExecutorService to use to schedule task.  If not specified
    then a single thread `scheduled-thread-pool` component is started when this component is started
    and stopped when this component is stopped.

  See *java.util.concurrent.ScheduledThreadPoolExecutor*"
  {:arglists '([task delay-duration & {:as opts :keys [initial-delay-duration executor-service]}])}
  ^ScheduledFuture [task delay-duration & {:as opts}]
  (map->ScheduledTaskComponent
    :task task
    :delay-duration delay-duration
    :make-scheduled-future-fn make-fixed-delay-scheduled-task
    opts))

(defn- proxy-thread-pool
  ^ThreadPoolExecutor
  [{:as opts :keys [core-pool-size max-pool-size queue convey-bindings? termination-wait-duration]}
   keep-alive-ns thread-factory rejected-execution-handler]
  (proxy [ThreadPoolExecutor Lifecycle]
    [core-pool-size max-pool-size keep-alive-ns TimeUnit/NANOSECONDS queue thread-factory
     rejected-execution-handler]
    (invokeAll
      ([tasks]
       (let [this ^ThreadPoolExecutor this]
         (proxy-super invokeAll (cond-> tasks convey-bindings? all-conveying-callables))))
      ([tasks timeout unit]
       (let [this ^ThreadPoolExecutor this]
         (proxy-super invokeAll (cond-> tasks convey-bindings? all-conveying-callables) timeout
           unit))))
    (invokeAny
      ([tasks]
       (let [this ^ThreadPoolExecutor this]
         (proxy-super invokeAny (cond-> tasks convey-bindings? all-conveying-callables))))
      ([tasks timeout unit]
       (let [this ^ThreadPoolExecutor this]
         (proxy-super invokeAny (cond-> tasks convey-bindings? all-conveying-callables) timeout
           unit))))
    (submit
      ([task]
       (let [this ^ThreadPoolExecutor this]
         (if (instance? Callable task)
           (let [task (cond-> task convey-bindings? conveying-callable)]
             (proxy-super submit ^Callable task))
           (let [task (cond-> task convey-bindings? conveying-runnable)]
             (proxy-super submit ^Runnable task)))))
      ([task result]
       (let [this ^ThreadPoolExecutor this]
         (proxy-super submit (cond-> task convey-bindings? conveying-runnable) result))))
    (execute
      [command]
      (let [this ^ThreadPoolExecutor this]
        (proxy-super execute (cond-> command convey-bindings? conveying-runnable))))
    (start
      []
      this)
    (stop
      []
      (let [this ^ThreadPoolExecutor this]
        (.shutdown this)
        (if-let [termination-wait-ns (some-> termination-wait-duration Duration/.toNanos)]
          (when (pos? termination-wait-ns)
            (.awaitTermination this termination-wait-ns TimeUnit/NANOSECONDS))
          (while (not (.isTerminated this)))))
      opts)))

(desiderata/defrecord ThreadPoolComponent
  [core-pool-size max-pool-size keep-alive-duration queue thread-factory rejected-execution-handler
   allow-core-thread-timeout? convey-bindings? termination-wait-duration]
  ::desiderata/defaults
  {:core-pool-size 1
   :max-pool-size 1
   :keep-alive-duration (Duration/ofMinutes 1)
   :convey-bindings? true
   :queue (LinkedBlockingQueue.)}
  component/Lifecycle
  (start
    [this]
    (let [keep-alive-ns (-> keep-alive-duration Duration/.toNanos)
          thread-factory (or thread-factory (Executors/defaultThreadFactory))
          rejected-execution-handler (adapt-rejected-execution-handler rejected-execution-handler)]
      (when (and convey-bindings? (caller-runs? rejected-execution-handler))
        (throw (IllegalStateException.
                 "convey-bindings? is not compatible with caller runs policy")))
      (cond-> (proxy-thread-pool this keep-alive-ns thread-factory rejected-execution-handler)
        allow-core-thread-timeout? (doto (.allowCoreThreadTimeOut true)))))
  (stop
    [this]
    this))

(defn- proxy-scheduled-thread-pool
  ^ScheduledThreadPoolExecutor
  [{:as opts :keys [core-pool-size convey-bindings? termination-wait-duration]} thread-factory
   rejected-execution-handler]
  (proxy [ScheduledThreadPoolExecutor Lifecycle]
    [core-pool-size thread-factory rejected-execution-handler]
    (invokeAll
      ([tasks]
       (let [this ^ScheduledThreadPoolExecutor this]
         (proxy-super invokeAll (cond-> tasks convey-bindings? all-conveying-callables))))
      ([tasks timeout unit]
       (let [this ^ScheduledThreadPoolExecutor this]
         (proxy-super invokeAll (cond-> tasks convey-bindings? all-conveying-callables) timeout
           unit))))
    (invokeAny
      ([tasks]
       (let [this ^ScheduledThreadPoolExecutor this]
         (proxy-super invokeAny (cond-> tasks convey-bindings? all-conveying-callables))))
      ([tasks timeout unit]
       (let [this ^ScheduledThreadPoolExecutor this]
         (proxy-super invokeAny (cond-> tasks convey-bindings? all-conveying-callables) timeout
           unit))))
    (submit
      ([task]
       (let [this ^ScheduledThreadPoolExecutor this]
         (if (instance? Callable task)
           (let [task (cond-> task convey-bindings? conveying-callable)]
             (proxy-super submit ^Callable task))
           (let [task (cond-> task convey-bindings? conveying-runnable)]
             (proxy-super submit ^Runnable task)))))
      ([task result]
       (let [this ^ScheduledThreadPoolExecutor this]
         (proxy-super submit (cond-> task convey-bindings? conveying-runnable) result))))
    (execute
      [command]
      (let [this ^ScheduledThreadPoolExecutor this]
        (proxy-super execute (cond-> command convey-bindings? conveying-runnable))))
    (schedule
      [command delay unit]
      (let [this ^ScheduledThreadPoolExecutor this]
        (if (instance? Callable command)
          (let [task (cond-> command convey-bindings? conveying-callable)]
            (proxy-super schedule ^Callable task ^long delay ^TimeUnit unit))
          (let [task (cond-> command convey-bindings? conveying-runnable)]
            (proxy-super schedule ^Runnable task ^long delay ^TimeUnit unit)))))
    (scheduleAtFixedRate
      [command initial-delay period unit]
      (let [this ^ScheduledThreadPoolExecutor this]
        (proxy-super scheduleAtFixedRate (cond-> command convey-bindings? conveying-runnable)
          initial-delay period unit)))
    (scheduleWithFixedDelay
      [command initial-delay delay unit]
      (let [this ^ScheduledThreadPoolExecutor this]
        (proxy-super scheduleAtFixedRate (cond-> command convey-bindings? conveying-runnable)
          initial-delay delay unit)))
    (start
      []
      this)
    (stop
      []
      (let [this ^ScheduledThreadPoolExecutor this]
        (.shutdown this)
        (if-let [termination-wait-ns (some-> termination-wait-duration Duration/.toNanos)]
          (when (pos? termination-wait-ns)
            (.awaitTermination this termination-wait-ns TimeUnit/NANOSECONDS))
          (while (not (.isTerminated this)))))
      opts)))

(desiderata/defrecord ScheduledThreadPoolComponent
  [core-pool-size thread-factory rejected-execution-handler remove-on-cancel?
   continue-existing-periodic-tasks-after-shutdown? execute-existing-delayed-tasks-after-shutdown?
   convey-bindings? termination-wait-duration]
  ::desiderata/defaults
  {:core-pool-size 1
   :convey-bindings? true}
  component/Lifecycle
  (start
    [this]
    (let [thread-factory (or thread-factory (Executors/defaultThreadFactory))
          rejected-execution-handler (adapt-rejected-execution-handler rejected-execution-handler)]
      (when (and convey-bindings? (caller-runs? rejected-execution-handler))
        (throw (IllegalStateException.
                 (str "convey-bindings? is not compatible with caller runs policy, "
                   "an error will occur"))))
      (when (and continue-existing-periodic-tasks-after-shutdown? (nil? termination-wait-duration))
        (throw (IllegalStateException.
                 (str "continue-existing-periodic-tasks-after-shutdown? is not compatible with nil"
                   " termination-wait-duration, a deadlock will occur"))))
      (cond-> (proxy-scheduled-thread-pool this thread-factory rejected-execution-handler)
        remove-on-cancel?
        (doto (.setRemoveOnCancelPolicy true))
        continue-existing-periodic-tasks-after-shutdown?
        (doto (.setContinueExistingPeriodicTasksAfterShutdownPolicy true))
        (false? execute-existing-delayed-tasks-after-shutdown?)
        (doto (.setExecuteExistingDelayedTasksAfterShutdownPolicy false)))))
  (stop
    [this]
    this))
