;; Copyright © technosophist
;;
;; This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of
;; the MPL was not distributed with this file, You can obtain one at http://mozilla.org/MPL/2.0/.
;;
;; This Source Code Form is "Incompatible With Secondary Licenses", as defined by the Mozilla Public
;; License, v. 2.0.
(ns systems.thoughtfull.amalgam
  (:refer-clojure :exclude [vector])
  (:require
    [com.stuartsierra.component :as component]
    [systems.thoughtfull.amalgam.executors :as executors]
    [systems.thoughtfull.amalgam.jdbc :as jdbc])
  (:import
    (java.time Duration)
    (java.util.concurrent ScheduledExecutorService TimeUnit)
    (javax.sql DataSource)))

(defn overwrite
  "Resolve current and configuration values by always taking the configuration value."
  [_v c]
  c)

(defn configure
  "Recursively merge config map into component using merge-fns as necessary.  Maps are recursively
  merged.  Sets, vectors and lists are non-recursively concatenated.

  This behavior can be overridden with merge-fns, which should match the structure of config and
  value.  A merge function takes two arguments: the current value, and the configured value
  (respectively) and should return the final, resolved value.

  Example:

  ```clojure
  user> (amalgam/configure {:a [1] :b {:c [2]}}
          {:a [10] :b {:c [20] :d 30}}
          {:b {:c amalgam/overwrite}})
  {:a [1 10], :b {:c [20], :d 30}}
  ```"
  ([component config] (configure component config {}))
  ([component config merge-fn]
   (if (fn? merge-fn)
     (merge-fn component config)
     (cond
       (and (map? config) (map? component))
       (reduce-kv
         (fn [m k c]
           (assoc m k
             (if (contains? m k)
               (configure (get m k) c (when (and (map? merge-fn) (contains? merge-fn k))
                                        (get merge-fn k)))
               c)))
         component
         config)
       (or (and (set? config) (set? component))
         (and (vector? config) (vector? component)))
       (into component config)
       (and (list? config) (list? component))
       (apply list (concat component config))
       :else
       config))))

(defn start-system
  "Construct, configure, then start a system.

  - **`make-system`** — no argument function returning a new system
  - **`read-config`** — no argument function returning the configuration map.  The configuration map
    should mirror the structure of the system.
  - **`merge-fns`** (optional) — map mirroring the structure of both the system and configuration
    maps that contains merge functions to resolve a final value from the current value and
    configured value.

  Example:

  ```clojure
  user> (amalgam/start-system #(component/system-map :component (map->Component {:foo :bar}))
          (constantly {:component {:foo :baz}})
          {:component {:foo amalgam/overwrite}})
  {:component {:foo :baz}}
  ```

  See [[configure]]"
  ([make-system read-config]
   (start-system make-system read-config {}))
  ([make-system read-config merge-fns]
   (-> (make-system)
     (configure (read-config) merge-fns)
     component/start)))

(defn run-system
  "Start a configured system, run it, and register a JVM shutdown hook to stop the system.  Blocks
  until the system stop.

  - **`make-system`** — no argument function returning a new system
  - **`read-config`** — no argument function returning the configuration map.  The configuration map
    should mirror the structure of the system.
  - **`merge-fns`** (optional) — map mirroring the structure of both the system and configuration
    maps that contains merge functions to resolve a final value from the current value and
    configured value.

  See [[configure]] [[start-system]]"
  ([make-system read-config]
   (run-system make-system read-config {}))
  ([make-system read-config merge-fns]
   (let [running? (promise)
         system (start-system make-system read-config merge-fns)]
     (.addShutdownHook (Runtime/getRuntime)
       (Thread.
         #(do (component/stop system)
            (deliver running? false))))
     (while (try @running? (catch InterruptedException _ false))))))

(defn component
  "Make a component that from an object by extending component/Lifecycle via metadata.

  The object must take matadata, and generally it is best as a map so it can be configured before
  starting(see [[configure]]).  The started component (returned from `start`) must also take
  metadata.

  The start and stop functions are optional, and if they are not given, then they default to
  no-ops.

  If a type is given, it will be set as the object's type (i.e. the `type` key of its metadata).

  After this component has been started, stopping it resets it back to `object`.  In other words,

  ```clojure
  (let [c (amalgam/component ...)]
    (assert (identical? c (component/stop (component/start c)))))
  ```

  - **`start`** (optional) — a single argument function taking the object and returning a started
    component, which must be able to take metadata.
  - **`stop`** (optional) — a single argument function taking the started component and releasing
    resources.  `stop` is side-effecting and its return value is ignored.
  - **`type`** (optional) — a symbol to set as the object's type (i.e. in the `:type` key of its
    metadata).

  Example:

  ```clojure
  (defn temp-file
    [& {:as opts :keys [path]}]
    (amalgam/component opts
      :start (fn [this] (assoc this :file (io/file (:path this))))
      :stop (fn [this] (io/delete-file (:file this) true))))
  (def my-system (component/system-map :scratch-file (temp-file :path \"/tmp/scratch\")))
  ```"
  [object & {:keys [start stop type]}]
  (cond-> (vary-meta object
            assoc
            `component/start
            (fn [this]
              (vary-meta (cond-> this start start)
                assoc
                `component/start
                (fn [this'] this')
                `component/stop
                (fn [this']
                  (when stop (stop this'))
                  this))))
    type (vary-meta assoc :type type)))

(defn ^:deprecated ^:no-doc make-component-fn
  "Make a component constructor that takes options as keyword args and returns a component.

  After this component has been started, stopping it resets it back to the options map.  In other
  words,

  ```clojure
  (let [make-component (make-component-fn ...)
        c (make-component ...)]
    (assert (identical? c (component/stop (component/start c)))))
  ```

  - **`start`** — (optional) a single argument function taking the options and returning a new
    instance of the component.  The return value of `start` should be able to take metadata.
  - **`stop`** — (optional) a single argument function taking the component and releasing resources.
    `stop` is side-effecting and its return value is ignored.

  Example:

  ```clojure
  (def temp-file
    (amalgam/make-component-fn
      :start (fn [this] (assoc this :file (io/file (:path this))))
      :stop (fn [this] (io/delete-file (:file this) true))))
  (def my-system (component/system-map :scratch-file (temp-file :path \"/tmp/scratch\")))
  ```"
  [& {:keys [start stop]}]
  (fn [& {:as opts}]
    (vary-meta opts
      assoc
      `component/start
      (fn [this]
        (vary-meta (cond-> this start start)
          assoc
          `component/start
          (fn [this'] this')
          `component/stop
          (fn [this']
            (when stop (stop this'))
            this))))))

(defn vector
  "A vector component that collects its dependencies into a vector.

  Example:

  ```clojure
  user> (-> (component/system-map :v (amalgam/vector :a :b) :a 1 :b 2) component/start :v)
  [1 2]
  ```"
  [& dependency-keys]
  (let [dependency-keys (vec dependency-keys)]
    (component/using
      (component {::dependency-keys (vec dependency-keys)}
        :type `VectorComponent
        :start (fn [{:as this ::keys [dependency-keys]}]
                 (into [] (map this) dependency-keys)))
      dependency-keys)))

(defn function*
  "A component that collects its configuration and dependencies into a map and when started passes
  them as the first argument to the given function.

  Example:

  ```clojure
  user> (defn do-something [deps c] (assoc deps :c c))
  #'user/do-something
  user> (def f (component/using (amalgam/function* do-something) [:a :b]))
  #'user/f
  user> (def system (-> (component/system-map :f f :a 1 :b 2) component/start))
  #'user/system
  user> ((:f system) 3)
  {:a 1, :b 2, :c 3}
  ```"
  [f]
  (component {}
    :type `FunctionComponent
    :start (fn [this] (partial f this))))

(defn- rewrite-arities
  [this-sym arities]
  (for [[bindings & body] arities]
    `(~(vec (next bindings))
      (let [~(first bindings) ~this-sym]
        ~@body))))

(defmacro function
  "A macro taking a fn-tail and making a component that collects its configuration and dependencies
  into a map and when started passes them as the first argument to a function made from the
  fn-tail.

  Example:

  ```clojure
  user> (def f (component/using (amalgam/function [deps c] (assoc deps :c c)) [:a :b]))
  #'user/f
  user> (def system (-> (component/system-map :f f :a 1 :b 2) component/start))
  #'user/system
  user> ((:f system) 3)
  {:a 1, :b 2, :c 3}
  ```"
  [& fn-tail]
  (let [ ;; normalize our fn-tail
        [_ name? & arities] (apply @#'fn nil nil fn-tail)
        this-sym (gensym)
        fn-tail (if (symbol? name?)
                  (cons name? (rewrite-arities this-sym arities))
                  (rewrite-arities this-sym (cons name? arities)))]
    `(component {}
       :type `FunctionComponent
       :start (fn [~this-sym] (fn* ~@fn-tail)))))

(defn thread-pool
  "Make a new thread pool component.  An unstarted thread pool component is a record that collects
  configuration (see [[configure]]).  Once started, it becomes a subclass of
  *java.util.concurrent.ThreadPoolExecutor* that also implements
  *com.stuartsierra.component.Lifecycle*.

  When the thread pool is stopped, it reverts back to the configuration record.  It can be started
  again with the same (or modified) configuration.

  Calling stop on the configuration record or start on the thread pool is idempotent, i.e., they
  return `this` unmodified.

  - **`:core-pool-size`** (optional) — number of threads to keep in the pool, even if they are idle,
    defaults to 1
  - **`:max-pool-size`** (optional) — maximum number of threads to allow in the pool, defaults to 1
  - **`:keep-alive-duration`** (optional) — when the number of threads is greater than the core, the
    maximum *java.time.Duration* a thread waits for new tasks before terminating, defaults to 1
    minute
  - **`:queue`** (optional) — queue to use for holding tasks before they are executed, defaults to a
    *java.util.concurrent.LinkedBlockingQueue* with *java.lang.Integer/MAX_VALUE* capacity
  - **`:thread-factory`** (optional) — factory to use when executor creates a new thread, defaults
    to `(Executors/defaultThreadFactory)`
  - **`:rejected-execution-handler`** (optional) — handler to use when execution is blocked because
    the thread and queue capacities are reached.  The handler can be an instance of
    *java.util.concurrent.RejectedExecutionHandler*, a function of two arguments (a *Runnable* and a
    *ThreadPoolExecutor*), or one of `:abort`, `:caller-runs`, `:discard-oldest`, `:discard`,
    defaults to `:abort`.
  - **`:allow-core-thread-timeout?`** (optional) — if true then allow core threads to timeout and
    terminate if no tasks arrive within the keep alive time, otherwise core threads never terminate
    due to lack of incoming tasks, defaults to false
  - **`:convey-bindings?`** (optional) — if true then convey thread local var bindings with tasks as
    they are submitted to the thread pool, defaults to true
  - **`:termination-wait-duration`** (optional) — a *java.time.Duration* to wait for the thread pool
    to terminate when the component is stopped.  A duration of zero means don't wait at all.  If not
    specified, then wait indefinitely."
  {:arglists '([& {:keys [core-pool-size max-pool-size keep-alive-duration queue thread-factory
                          rejected-execution-handler convey-bindings?
                          termination-wait-duration]}])}
  [& {:as opts}]
  (executors/map->ThreadPoolComponent opts))

(defn scheduled-thread-pool
  "Make a new scheduled thread pool.  An unstarted scheduled thread pool is a record that collects
  configuration (see [[configure]]).  Once started, it becomes a subclass of
  *java.util.concurrent.ScheduledThreadPoolExecutor* that also implements
  *com.stuartsierra.component.Lifecycle*.

  Wher the scheduled thread pool is stopped, it reverts back to the configuration record.  It can
  be started again with the same (or modified) configuration.

  Calling stop on the configuration record or start on the scheduled thread pool is idempotent,
  i.e., they return `this` unmodified.

  - **`:core-pool-size`** (optional) — number of threads to use to run scheduled tasks, defaults to
    1
  - **`:thread-factory`** (optional) — factory to use when executor creates a new thread, defaults
    to `(Executors/defaultThreadFactory)`
  - **`:rejected-execution-handler`** (optional) — handler to use when execution is blocked because
    the thread and queue capacities are reached.  The handler can be an instance of
    *java.util.concurrent.RejectedExecutionHandler*, a function of two arguments (a *Runnable* and a
    *ThreadPoolExecutor*), or one of `:abort`, `:caller-runs`, `:discard-oldest`, `:discard`,
    defaults to `:abort`.
  - **`:remove-on-cancel?`** (optional) — if true immediately remove canceled tasks from the queue,
    otherwise wait until their delay elapses, defaults to false
  - **`:continue-existing-periodic-tasks-after-shutdown?`** (optional) — if true continue executing
    existing periodic tasks even when this executor has been shutdown, otherwise immediately stop
    them, defaults to false
  - **`:execute-existing-delayed-tasks-after-shutdown?`** (optional) — if true execute existing
    delayed tasks even when this executor has been shutdown, otherwise immediately cancel them,
    defaults to true
  - **`:convey-bindings?`** (optional) — if true then convey thread local var bindings with tasks as
    they are submitted to the thread pool, defaults to true
  - **`:termination-wait-duration`** (optional) — a *java.time.Duration* to wait for the thread pool
    to terminate when the component is stopped.  A duration of zero means don't wait at all.  If not
    specified, then wait indefinitely."
  {:arglists '([& {:keys [core-pool-size thread-factory rejected-execution-handler remove-on-cancel?
                          continue-existing-periodic-tasks-after-shutdown?
                          execute-existing-delayed-tasks-after-shutdown? convey-bindings?
                          termination-wait-duration]}])}
  [& {:as opts}]
  (executors/map->ScheduledThreadPoolComponent opts))

(defn fixed-rate-scheduled-task
  "Make a new fixed rate ScheduledTaskComponent wrapping a ScheduledFuture.

  - **`:task`** — a no argument function to run as a scheduled task.
  - **`:period-duration`** — *java.time.Duration* from the start of one run to the start of the next
  - **`:initial-delay-duration`** (optional) — *java.time.Duration* to wait before the first run of
    `task`, defaults to 0
  - **`:scheduled-executor-service`** (optional) — ScheduledExecutorService to use to schedule task.
    If not specified then a single thread `scheduled-thread-pool` component is started when this
    component is started and stopped when this component is stopped.

  See *java.util.concurrent.ScheduledThreadPoolExecutor*"
  {:arglists '([& {:keys [task period-duration initial-delay-duration
                          scheduled-executor-service]}])}
  [& {:as opts}]
  (executors/map->ScheduledTaskComponent
    :make-scheduled-future-fn
    (fn [^ScheduledExecutorService scheduled-executor-service
         {:keys [task period-duration initial-delay-duration]}]
      (when (nil? task)
        (throw (IllegalStateException. "task is required")))
      (when (nil? period-duration)
        (throw (IllegalStateException. "period-duration is required")))
      (let [initial-delay-ns (or (some-> initial-delay-duration Duration/.toNanos) 0)
            period-ns (some-> period-duration Duration/.toNanos)]
        (.scheduleAtFixedRate scheduled-executor-service task initial-delay-ns period-ns
          TimeUnit/NANOSECONDS)))
    opts))

(defn fixed-delay-scheduled-task
  "Make a new fixed delay ScheduledTaskComponent wrapping a ScheduledFuture.

  - **`:task`** — a no argument function to run as a scheduled task.
  - **`:delay-duration`** — *java.time.Duration* from the end of one run to the start of the next
  - **`:initial-delay-duration`** (optional) — *java.time.Duration* to wait before the first run of
    `task`, defaults to 0
  - **`:scheduled-executor-service`** (optional) — ScheduledExecutorService to use to schedule task.
    If not specified then a single thread `scheduled-thread-pool` component is started when this
    component is started and stopped when this component is stopped.

  See *java.util.concurrent.ScheduledThreadPoolExecutor*"
  {:arglists '([& {:keys [task delay-duration initial-delay-duration scheduled-executor-service]}])}
  [& {:as opts}]
  (executors/map->ScheduledTaskComponent
    :make-scheduled-future-fn
    (fn [^ScheduledExecutorService scheduled-executor-service
         {:keys [task delay-duration initial-delay-duration]}]
      (when (nil? task)
        (throw (IllegalStateException. "task is required")))
      (when (nil? delay-duration)
        (throw (IllegalStateException. "delay-duration is required")))
      (let [initial-delay-ns (or (some-> initial-delay-duration Duration/.toNanos) 0)
            delay-ns (some-> delay-duration Duration/.toNanos)]
        (.scheduleWithFixedDelay scheduled-executor-service task initial-delay-ns delay-ns
          TimeUnit/NANOSECONDS)))
    opts))

(defn data-source
  "Make a new DataSourceComponent wrapping a javax.sql.DataSource.

  If the wrapped data source implements *java.io.AutoCloseable* or *java.io.Closeable*, then when
  the component is stopped it will close the data source.  If the wrapped data source does not
  implement either interface, then stop will not attempt to close it.

  - **`make-data-source-fn`** — a function that takes the DataSourceComponent as an argument and
    creates a DataSource for wrapping.  Any options necessary for constructing a DataSource should
    be taken from the DataSourceComponent.
  - **`login-timeout-duration`** (optional) — the maximum *java.time.Duration* to wait while
    attempting to connect to a database.  A value of zero specifies that the timeout is the default
    system timeout if there is one; otherwise, it specifies that there is no timeout. When a
    DataSource object is created, the login timeout is initially zero.
  - **`log-writer`** (optional) — a *java.io.PrintWriter* to set as the log writer for this data
    source, defaults to nil which disables logging.

  Any additional options are passed along to the DataSourceComponent and then to
  `make-data-source-fn` when the component is started.

  See *javax.sql.DataSource*"
  {:arglists '([& {:keys [make-data-source-fn login-timeout-duration log-writer]}])}
  ^DataSource [& {:as opts}]
  (jdbc/map->DataSourceComponent opts))
