;; Copyright © technosophist
;;
;; This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of
;; the MPL was not distributed with this file, You can obtain one at http://mozilla.org/MPL/2.0/.
;;
;; This Source Code Form is "Incompatible With Secondary Licenses", as defined by the Mozilla Public
;; License, v. 2.0.
(ns systems.thoughtfull.amalgam.dirigiste
  (:require
    [com.stuartsierra.component :as component]
    [systems.thoughtfull.amalgam.executors :as executors]
    [systems.thoughtfull.desiderata :as desiderata])
  (:import
    (com.stuartsierra.component Lifecycle)
    (io.aleph.dirigiste Executor Executors Stats Stats$Metric)
    (java.time Duration)
    (java.util EnumSet)
    (java.util.concurrent SynchronousQueue TimeUnit)))

(set! *warn-on-reflection* true)

(def ^:private ->metric
  {:queue-length Stats$Metric/QUEUE_LENGTH
   :queue-latency Stats$Metric/QUEUE_LATENCY,
   :task-latency Stats$Metric/TASK_LATENCY,
   :task-arrival-rate Stats$Metric/TASK_ARRIVAL_RATE,
   :task-completion-rate Stats$Metric/TASK_COMPLETION_RATE,
   :task-rejection-rate Stats$Metric/TASK_REJECTION_RATE,
   :utilization Stats$Metric/UTILIZATION})

(defn- proxy-executor
  ^Executor
  [{:as opts
    :keys [initial-thread-count sample-duration control-duration convey-bindings?
           termination-wait-duration]}
   thread-factory queue controller metrics-set]
  (proxy [Executor Lifecycle]
    [thread-factory queue controller initial-thread-count metrics-set
     (Duration/.toMillis sample-duration) (Duration/.toMillis control-duration)
     TimeUnit/MILLISECONDS]
    (invokeAll
      ([tasks]
       (let [this ^Executor this]
         (proxy-super invokeAll (cond-> tasks convey-bindings? executors/all-conveying-callables))))
      ([tasks timeout unit]
       (let [this ^Executor this]
         (proxy-super invokeAll (cond-> tasks convey-bindings? executors/all-conveying-callables)
           timeout unit))))
    (invokeAny
      ([tasks]
       (let [this ^Executor this]
         (proxy-super invokeAny (cond-> tasks convey-bindings? executors/all-conveying-callables))))
      ([tasks timeout unit]
       (let [this ^Executor this]
         (proxy-super invokeAny (cond-> tasks convey-bindings? executors/all-conveying-callables)
           timeout unit))))
    (submit
      ([task]
       (let [this ^Executor this]
         (if (instance? Callable task)
           (let [task (cond-> task convey-bindings? executors/conveying-callable)]
             (proxy-super submit ^Callable task))
           (let [task (cond-> task convey-bindings? executors/conveying-runnable)]
             (proxy-super submit ^Runnable task)))))
      ([task result]
       (let [this ^Executor this]
         (proxy-super submit (cond-> task convey-bindings? executors/conveying-runnable) result))))
    (execute
      [command]
      (let [this ^Executor this]
        (proxy-super execute (cond-> command convey-bindings? executors/conveying-runnable))))
    (executeWithoutRejection
      [task]
      (let [this ^Executor this]
        (proxy-super executeWithoutRejection
          (cond-> task convey-bindings? executors/conveying-runnable))))
    (start
      []
      this)
    (stop
      []
      (let [this ^Executor this]
        (.shutdown this)
        (if-let [termination-wait-ns (some-> termination-wait-duration Duration/.toNanos)]
          (when (pos? termination-wait-ns)
            (.awaitTermination this termination-wait-ns TimeUnit/NANOSECONDS))
          (while (not (.isTerminated this)))))
      opts)))

(desiderata/defrecord ExecutorComponent
  [thread-factory queue controller target-utilization-percent max-thread-count initial-thread-count
   metrics sample-duration control-duration]
  ::desiderata/defaults
  {:queue (SynchronousQueue. false)
   :initial-thread-count 1
   :metrics #{}
   :sample-duration (Duration/ofMillis 25)
   :control-duration (Duration/ofSeconds 1)
   :convey-bindings? true}
  component/Lifecycle
  (start
    [this]
    (let [thread-factory (or thread-factory
                           (desiderata/thread-factory
                             :name "dirigiste"
                             :convey-bindings? false
                             :daemon? true))
          queue (or (:blocking-queue this) queue)
          utilization? (and target-utilization-percent max-thread-count)
          controller (or controller
                       (when utilization?
                         (Executors/utilizationController target-utilization-percent
                           max-thread-count))
                       (Executors/fixedController initial-thread-count))
          metrics (cond-> metrics utilization? (conj :utilization))
          metrics-set (EnumSet/noneOf Stats$Metric)]
      (doseq [metric metrics]
        (when-not (contains? ->metric metric)
          (throw (IllegalArgumentException. (str "Invalid metric " metric))))
        (.add metrics-set (->metric metric)))
      (proxy-executor this thread-factory queue controller metrics-set)))
  (stop
    [this]
    this))

(defn thread-pool
  "Make a new self-adjusting thread pool component with metrics reporting.  An unstarted thread pool
  component is a record the collects configuration.  Once started, it becomes a subclass of
  *io.aleph.dirigiste.Executor* that also implements *com.stuartsierra.component.Lifecycle*.

  When the thread pool is stopped, it reverts back to the configuration record.  It can be started
  again with the same (or modified) configuration.

  Being a *io.aleph.dirigiste.Executor*, its controller determines whether to add or remove
  threads.  The default controller is a fixed controller with an initial thread count that never
  increases or decreases.

  If you specify a target utilization percentage and maximum thread count, then a utilization
  controller will decide when to add or remove threads.  It will never allow more than the maximum
  number of thread, and it will adjust when the current pool utilization is above or below the
  target utilization.

  Otherwise, you can also specify your own controller, which it will use and ignore other
  controller related options.

  Also being a dirigiste *Executor*, it collects statistics about the thread pool, which you can
  retrieve using the stats function.  If the controller is a utilization controller, then the
  component ensures utilization is a metric the control loop collects.

  A blocking queue sits in front of the thread pool.  By default it is an unfair synchronous queue.
  You can provide your own blocking queue instance.

  Even though it is possible to shut it down before stopping it, preferably you would just stop
  the component.

  - **`thread-factory`** (optional) — factory used to create threads when adding to the pool,
    defaults to new desiderata daemon thread factory.
  - **`blocking-queue`** (optional) — task queue, defaults to an unfair
    *java.util.concurent.SynchronousQueue*.
  - **`target-utilization-percent`** (optional) — target utilization percentage (float between 0.0
    and 1.0) for thread pool.
  - **`max-thread-count`** (optional) — maximum number of threads allowed.
  - **`controller`** (optional) — controller for adding/removing threads.
  - **`initial-thread-count`** (optional) — number of threads initially in the pool, defaults to 1.
  - **`metrics`** (optional) — set of keywords for metrics about which to collect statistics,
    defaults to empty set, valid metrics are `:queue-length`, `:queue-latency`, `:task-latency`,
    `:task-arrival-rate`, `:task-completion-rate`, `:task-rejection-rate`, `:utilization`.
  - **`sample-duration`** (optional) — *java.time.Duration* between stats samples, defaults to 25
    milliseconds.
  - **`control-duration`** (optional) — *java.time.Duration* between control decisions, defaults to
    1 second.
  - **`convey-bindings?`** (optional) — if true then convey thread local var bindings with tasks as
    they are submitted to the Executor, defaults to true.
  - **`termination-wait-duration`** (optional) — *java.time.Duration* to wait for shutdown, if zero
    do not await termination, if not specified, wait indefinitely."
  {:arglists
   '([& {:keys [thread-factory blocking-queue target-utilization-percent max-thread-count
                controller initial-thread-count metrics sample-duration control-duration
                convey-bindings? termination-wait-duration]}])}
  [& {:as opts}]
  (map->ExecutorComponent opts))

(defn stats
  "Stats for dirigiste thread pool.

  - **`quantiles`** — a map from metric to a map from labels to quantiles (e.g. `{:queue-latency
    {:q90 0.9}}`), quantiles are the points within the distribution to look up, 0.5 returns the
    median, 0.9 the 90th percentile.

  Returns a map with the following keys:

  - **`num-workers`** — number of active workers in the pool.
  - **`queue-latency`** — queue latency as double value of nanoseconds.
  - **`queue-length`** — queue length as double value.
  - **`task-arrival-rate`** — task arrival rate as double value of tasks per second.
  - **`task-completion-rate`** — task completion rate as double value of tasks per second.
  - **`task-latency`** — task latency as double value of nanoseconds.
  - **`task-rejection-rate`** — task rejection rate as double value of tasks per second.
  - **`utilization`** — utilization of workers as a double value between 0 and 1.

  Except for `:num-workers` which is simply an integer value, the value at each key is a map with
  a `:mean` for the metric's mean value and the labels for each quantile.

  Example:

  ```clojure
  user> (stats executor-service :queue-latency {:p90 0.9})
  {:num-workers 11,
   :queue-latency {:mean 93243.0, :p90 118024.5},
   :queue-length {:mean 0.0},
   :task-arrival-rate {:mean 10.0},
   :task-completion-rate {:mean 10.0},
   :task-latency {:mean 1.001146684E9},
   :task-rejection-rate {:mean 0.0},
   :utilization {:mean 0.9017880571953183}}
  ```"
  [thread-pool & {:as quantiles}]
  (let [stats (Executor/.getLastStats thread-pool)
        collect (fn [s metric [mean-fn quantile-fn]]
                  (let [qq (metric quantiles)
                        mm {:mean (mean-fn stats)}]
                    (assoc! s metric
                      (if (seq qq)
                        (persistent!
                          (reduce-kv (fn [s l q] (assoc! s l (quantile-fn stats q)))
                            (transient mm)
                            qq))
                        mm))))]
    (persistent!
      (reduce-kv collect
        (transient {:num-workers (.getNumWorkers stats)})
        {:queue-latency [Stats/.getMeanQueueLatency Stats/.getQueueLatency]
         :queue-length [Stats/.getMeanQueueLength Stats/.getQueueLength]
         :task-arrival-rate [Stats/.getMeanTaskArrivalRate Stats/.getTaskArrivalRate]
         :task-completion-rate [Stats/.getMeanTaskCompletionRate Stats/.getTaskCompletionRate]
         :task-latency [Stats/.getMeanTaskLatency Stats/.getTaskLatency]
         :task-rejection-rate [Stats/.getMeanTaskRejectionRate Stats/.getTaskRejectionRate]
         :utilization [Stats/.getMeanUtilization Stats/.getUtilization]}))))
