;; Copyright © technosophist
;;
;; This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of
;; the MPL was not distributed with this file, You can obtain one at http://mozilla.org/MPL/2.0/.
;;
;; This Source Code Form is "Incompatible With Secondary Licenses", as defined by the Mozilla Public
;; License, v. 2.0.
(ns systems.thoughtfull.amalgam.dirigiste
  (:refer-clojure :exclude [defrecord])
  (:require
    [systems.thoughtfull.amalgam.executors :as executors]
    [systems.thoughtfull.desiderata :as desiderata])
  (:import
    (io.aleph.dirigiste Executor Executors Stats Stats$Metric)
    (java.time Duration)
    (java.util EnumSet)
    (java.util.concurrent SynchronousQueue TimeUnit)))

(set! *warn-on-reflection* true)

(def ^:private ->metric
  {:queue-length Stats$Metric/QUEUE_LENGTH
   :queue-latency Stats$Metric/QUEUE_LATENCY,
   :task-latency Stats$Metric/TASK_LATENCY,
   :task-arrival-rate Stats$Metric/TASK_ARRIVAL_RATE,
   :task-completion-rate Stats$Metric/TASK_COMPLETION_RATE,
   :task-rejection-rate Stats$Metric/TASK_REJECTION_RATE,
   :utilization Stats$Metric/UTILIZATION})

(defn- make-dirigiste-executor
  [& {:keys [thread-factory blocking-queue controller target-utilization-percent max-thread-count
             initial-thread-count metrics sample-duration control-duration]}]
  (let [blocking-queue (or blocking-queue (SynchronousQueue. false))
        metrics (or #{} metrics)
        sample-duration (Duration/ofMillis (or sample-duration 25))
        control-duration (Duration/ofSeconds (or control-duration 1))
        thread-factory (or thread-factory
                         (desiderata/thread-factory
                           :name "dirigiste"
                           :convey-bindings? false
                           :daemon? true))
        utilization? (and target-utilization-percent max-thread-count)
        controller (or controller
                     (when utilization?
                       (Executors/utilizationController target-utilization-percent
                         max-thread-count))
                     (Executors/fixedController initial-thread-count))
        metrics (cond-> metrics utilization? (conj :utilization))]
    (doseq [metric metrics]
      (when-not (contains? ->metric metric)
        (throw (IllegalArgumentException. (str "Invalid metric " metric)))))
    (let [metrics-set (EnumSet/noneOf Stats$Metric)]
      (doseq [metric metrics]
        (.add metrics-set (->metric metric)))
      (Executor. thread-factory blocking-queue controller initial-thread-count metrics-set
        (Duration/.toMillis sample-duration) (Duration/.toMillis control-duration)
        TimeUnit/MILLISECONDS))))

(defn thread-pool
  "Self-adjusting thread pool component with metrics reporting.

  Being a *io.aleph.dirigiste.Executor*, its controller determines whether to add or remove
  threads.  The default controller is a fixed controller with an initial thread count that never
  increases or decreases.

  If you specify a target utilization percentage and maximum thread count, then a utilization
  controller will decide when to add or remove threads.  It will never allow more than the maximum
  number of thread, and it will adjust when the current pool utilization is above or below the
  target utilization.

  Otherwise, you can also specify your own controller, which it will use and ignore other
  controller related options.

  Also being a dirigiste *Executor*, it collects metrics about the thread pool, which you can
  retrieve using the stats function.  If the controller is a utilization controller, then the
  component ensures utilization is a metric the control loop collects.

  A blocking queue sits in front of the executor.  By default it is an unfair synchronous queue.
  You can provide your own blocking queue implementation.

  Even though it is possible to shut it down before stopping it, preferably you would just stop
  the component.

  - **`initial-thread-count`** — required number of threads initially in the pool.
  - **`thread-factory`** (optional) — factory used to create threads when adding to the pool,
    defaults to new desiderata daemon thread factory.
  - **`blocking-queue`** (optional) — task queue, defaults to an unfair
    *java.util.concurent.SynchronousQueue*.
  - **`target-utilization-percent`** (optional) — target utilization percentage for thread pool.
  - **`max-thread-count`** (optional) — maximum number of threads allowed.
  - **`controller`** (optional) — controller for adding/removing threads.
  - **`metrics`** (optional) — set of keywords for metrics to collect, defaults to empty set,
    valid metrics are :queue-length, :queue-latency, :task-latency, :task-arrival-rate,
    :task-completion-rate, :task-rejection-rate, :utilization.
  - **`sample-duration`** (optional) — java.time.Duration between stats samples, defaults to 25
    milliseconds.
  - **`control-duration`** (optional) — java.time.Duration between control decisions, defaults to
    1 second.
  - **`convey-bindings?`** (optional) — if true then convey thread local var bindings with tasks as
    they are submitted to the wrapped ExecutorService, defaults to true.
  - **`termination-wait-duration`** (optional) — java.time.Duration to wait for shutdown, if zero do
    not await termination, if not specified, wait indefinitely.

  For details about **`convey-bindings?`** and **`termination-wait-duration`** see
  `systems.thoughtfull.amalgam/ExecutorServiceComponent`"
  {:arglists '([initial-thread-count & {:keys [thread-factory blocking-queue
                                               target-utilization-precent max-thread-count
                                               controller metrics sample-duration
                                               control-duration convey-bindings?
                                               termination-wait-duration]}])}
  [initial-thread-count & {:as opts :keys [convey-bindings? termination-wait-duration]}]
  (executors/map->ExecutorServiceComponent
    :initial-thread-count initial-thread-count
    :make-executor-service-fn make-dirigiste-executor
    :convey-bindings? convey-bindings?
    :termination-wait-duration termination-wait-duration
    opts))

(defn execute-without-rejection
  "A version of execute which will simply block until the task is accepted, rather than throwing a
  *RejectedExceptionException*.  *RejectedExecutionException* will only be thrown if the executor
  is shut down.

  If thread bindings are established at the time the task is queued, then the task is wrapped to
  convey the bindings, otherwise the task is queued unmodified.

  Returns nil."
  [executor-service ^Runnable task]
  (let [executor ^Executor (executors/executor-service executor-service)
        convey-bindings? (:convey-bindings? executor-service)]
    (.executeWithoutRejection executor
      (if convey-bindings? (executors/conveying-runnable task) task))))

(defn stats
  "Stats for dirigiste Executor component.

  - **`quantiles`** — a map from metric to a map from labels to quantiles (e.g. `{:queue-latency
    {:q90 0.9}}`), quantiles are the points within the distribution to look up, 0.5 returns the
    median, 0.9 the 90th percentile.

  Returns a map with the following keys:

  - **`num-workers`** — number of active workers in the pool.
  - **`queue-latency`** — queue latency as double value of nanoseconds.
  - **`queue-length`** — queue length as double value.
  - **`task-arrival-rate`** — task arrival rate as double value of tasks per second.
  - **`task-completion-rate`** — task completion rate as double value of tasks per second.
  - **`task-latency`** — task latency as double value of nanoseconds.
  - **`task-rejection-rate`** — task rejection rate as double value of tasks per second.
  - **`utilization`** — utilization of workers as a double value between 0 and 1.

  Except for :num-workers which is simply an integer value, the value at each key is a map with
  a :mean for the metric's mean value and the labels for each quantile.

  Example:

  ```clojure
  user> (stats executor-service :queue-latency {:p90 0.9})
  {:num-workers 11,
   :queue-latency {:mean 93243.0, :p90 118024.5},
   :queue-length {:mean 0.0},
   :task-arrival-rate {:mean 10.0},
   :task-completion-rate {:mean 10.0},
   :task-latency {:mean 1.001146684E9},
   :task-rejection-rate {:mean 0.0},
   :utilization {:mean 0.9017880571953183}}
  ```"
  [executor-service & {:as quantiles}]
  (let [executor ^Executor (executors/executor-service executor-service)
        stats (.getLastStats executor)
        collect (fn [s metric [mean-fn quantile-fn]]
                  (let [qq (metric quantiles)
                        mm {:mean (mean-fn stats)}]
                    (assoc! s metric
                      (if (seq qq)
                        (persistent!
                          (reduce-kv (fn [s l q] (assoc! s l (quantile-fn stats q)))
                            (transient mm)
                            qq))
                        mm))))]
    (persistent!
      (reduce-kv collect
        (transient {:num-workers (.getNumWorkers stats)})
        {:queue-latency [Stats/.getMeanQueueLatency Stats/.getQueueLatency]
         :queue-length [Stats/.getMeanQueueLength Stats/.getQueueLength]
         :task-arrival-rate [Stats/.getMeanTaskArrivalRate Stats/.getTaskArrivalRate]
         :task-completion-rate [Stats/.getMeanTaskCompletionRate Stats/.getTaskCompletionRate]
         :task-latency [Stats/.getMeanTaskLatency Stats/.getTaskLatency]
         :task-rejection-rate [Stats/.getMeanTaskRejectionRate Stats/.getTaskRejectionRate]
         :utilization [Stats/.getMeanUtilization Stats/.getUtilization]}))))
