(ns symmetry.lib.atomfile
  (:require [clojure.java.io :as io]
            [clojure.tools.logging :as log]
            ;;[taoensso.nippy :as nippy]
            [fipp.edn :as fipp])
  (:import org.apache.commons.io.FileUtils))

(defn atomfile [path initvalue value->bytes bytes->value]
  (.mkdirs (.getParentFile (io/file path)))
  (let [read (fn []
               (-> (io/file path)
                   (FileUtils/readFileToByteArray)
                   (bytes->value)))
        write! (fn [v]
                 (with-open [out (java.io.FileOutputStream.
                                  (io/file (str path ".tmp")))]
                   (.write out (value->bytes v))
                   (.flush out))
                 (assert (.renameTo (io/file (str path ".tmp")) (io/file path))))
        cache (atom (if (.exists (io/file path))
                      (read)
                      initvalue))
        writer (agent {:last-v nil}
                      :error-mode :continue
                      :error-handler (fn [a ex]
                                       (log/error ex "atomfile write error")))]
    (add-watch cache :writer (fn [_ _ _ _]
                               (send-off writer
                                 (fn [state]
                                   (let [v @cache]
                                     (if (= v (:last-v state))
                                       state
                                       (do (write! v)
                                           (assoc state :last-v v))))))))
    (swap! cache identity) ;; force initial write
    cache))

;; (defn nippy-atomfile [path initvalue]
;;   (atomfile path
;;             initvalue
;;             nippy/freeze
;;             nippy/thaw))

(defn edn-atomfile [path initvalue]
  (atomfile path
            initvalue
            #(.getBytes (pr-str %))
            #(read-string (String. %))))

(defn fipp-atomfile [path initvalue]
  (atomfile path
            initvalue
            #(.getBytes (with-out-str (fipp/pprint %)))
            #(read-string (String. %))))

(comment

  (def A (edn-atomfile "/tmp/testfile.clj" {}))
  (println (count (keys @A)))
  (doseq [i (range 10000)]
    (swap! A assoc i (* i i)))

  ;; (def B (nippy-atomfile "/tmp/testfile.nip" {}))
  ;; (println (count (keys @B)))
  ;; (doseq [i (range 10000)]
  ;;   (swap! B assoc i (* i i)))

  (def code (fipp-atomfile "/tmp/testcode.clj"
                           {:code (read-string
                                   (slurp "src/symmetry/lib/atomfile.clj"))}))


  )
