(ns swing.request.index
  "Functions that deal with indexing requests appropriately"
  #? (:clj (:require [clj-time.core :as t]
                     [clj-time.coerce :as tc]
                     [datomic.api :as d]))
  #? (:cljs (:require [cljs-time.core :as t]
                      [cljs-time.coerce :as tc]
                      [datascript :as d])))

(defn start-time
  "Given value of :sub-req/time, return job start date-time"
  [times]
  (->> times
       (map :interval/start)
       (remove nil?)
       sort
       first
       tc/to-date-time))

(defn end-time
  "Given value of :sub-req/time, return job end date-time"
  [times]
  (->> times
       (map :interval/end)
       (remove nil?)
       sort
       last
       tc/to-date-time))

(defn index-time-tx
  [{indexed-start :sub-req.index/start
    indexed-end :sub-req.index/end
    time :sub-req/time
    req-db-id :db/id
    :as req}]
  (concat
   (let [start (tc/to-date (start-time time))]
     (if (and start (not= start indexed-start))
       [[:db/add req-db-id :sub-req.index/start start]]))
   (let [end (tc/to-date (end-time time))]
     (if (and end (not= end indexed-end))
       [[:db/add req-db-id :sub-req.index/end end]]))))

(defn with-index
  "Return a map that contains a DB with indexed columns added to the
  given request and the request itself. Note that the single argument
  form only returns the new entity itself."
  ([req]
   (->> (index-time-tx req)
        (reduce (fn [memo [add-retract e a v]]
                  (merge memo (if add-retract {a v} {})))
                req)))
  ([db req]
   (let [indexed-db (:db-after (d/with db (index-time-tx req)))
         indexed-entity (d/entity indexed-db (:db/id req))]
     {:db indexed-db :entity indexed-entity})))

(def ^:dynamic *general-pool-offset* (t/hours 24))

(defn index-general-pool-notify-tx
  [{{preferred-subs :sub-req/preferred-subs
     notify-time :sub-req.index/general-pool-notify-time
     req-db-id :db/id
     :as sub-req} :event/sub-req
    event-db-id :db/id
    create-time :db/txInstant
    :as event}]
  ;; NOTE: This means that if you sepcify a notify-time we will not
  ;; overwrite it. In theory this allows for custom general pool
  ;; notification times, with the below being the default behavior if
  ;; not specified.
  (if (not notify-time)
    [[:db/add req-db-id :sub-req.index/general-pool-notify-time (tc/to-date (t/plus (tc/to-date-time create-time) *general-pool-offset*))]]
    []))
