(ns sweet-tooth.workflow.tasks
  {:boot/export-tasks true}
  (:require [adzerk
             [boot-cljs :refer [cljs]]
             [boot-reload :refer [reload]]]
            [boot
             [core :as c :refer [deftask with-pre-wrap with-post-wrap]]
             [pod :as pod]
             [tmpdir :as tmpd]]
            [boot.task.built-in :refer [aot jar pom repl target uber watch]]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [deraen.boot-sass :as bs]
            [integrant.repl :as ir]
            [samestep.boot-refresh :refer [refresh]]
            [cider.tasks :as cider]))

(defn sym->var
  [sym]
  (if (symbol? sym)
    (try (let [ns-sym (-> sym namespace symbol)]
           (require ns-sym)
           (-> ns-sym find-ns (ns-resolve sym)))
         (catch java.lang.NullPointerException e
           (throw (Exception. "Could not find var " sym))))
    sym))

;; simple expire
(defn- expire-rewrite
  [index]
  (let [timestamp (quot (System/currentTimeMillis) 1000)]
    (str/replace index #"main.(css|js)" (str "main.$1?at=" timestamp))))

(deftask expire-assets
  "modify index.html to change URLs to expire assets"
  []
  (let [dir (c/tmp-dir!)]
    (with-pre-wrap fileset
      (let [index (first (c/by-path #{"index.html"} (c/input-files fileset)))
            file  (io/file dir "index.html")]
        (if (.exists (c/tmp-file index))
          (spit file (expire-rewrite (slurp (c/tmp-file index)))))
        (c/commit! (c/add-resource fileset dir))))))

(deftask build
  "Builds an uberjar"
  [v version VERSION str "version number"
   p project PROJECT sym "project name"
   m main    MAIN    sym "server ns with main fn"
   f file    FILE    str "name of jar file"]
  [version project main file]
  (c/merge-env! :resource-paths (c/get-env :source-paths))
  (comp (bs/sass)
        (cljs :optimizations :advanced
              :compiler-options {:parallel-build true})
        (expire-assets)
        (pom :project project :version version)
        (uber :exclude (conj pod/standard-jar-exclusions #".*\.html" #"license" #"LICENSE")) ; needed for arcane magic reasons
        (aot :namespace #{main})
        (jar :main main :file file :project project)
        (target :dir #{"target/build"})))

(deftask reload-integrant
  "Suspends integrant system before next task, and resumes integrant
  after."
  [p prep-fn PREP-FN sym "Name of function that preps the duct config"]
  (ir/set-prep! (sym->var prep-fn))
  (comp (with-pre-wrap fileset
          (ir/suspend)
          fileset)
        (with-post-wrap _
          (ir/resume))))

(defn guard-task
  "Only execute a task if files changed match a regex"
  [regexes task]
  (let [prev-before (atom nil)
        prev-after  (atom nil)
        task-next (fn [before-fileset]
                    (fn [after-fileset]
                      (reset! prev-after (c/fileset-diff before-fileset after-fileset))
                      after-fileset))
        apply-prev-after (fn [fileset]
                           (let [prev-after @prev-after
                                 remove (tmpd/removed fileset prev-after)
                                 added (tmpd/added fileset prev-after)
                                 changed (tmpd/changed fileset prev-after)]
                             (-> fileset
                                 (update :dirs clojure.set/union (:dirs added) (:dirs changed))
                                 (update :tree merge (:tree added) (:tree changed))
                                 c/commit!)))]
    (fn [next-handler]
      (fn [fileset]
        (let [diff (c/fileset-diff @prev-before fileset :hash)]
          (reset! prev-before fileset)
          ;; if the tree is empty no source file changed, something
          ;; else triggered loop. like a checkout changing.
          (if (or (empty? (:tree diff))
                  (seq (c/by-re regexes (c/input-files diff))))
            (next-handler ((task (task-next fileset)) fileset))
            (next-handler (apply-prev-after fileset))))))))

(deftask dev
  [R no-repl       bool            "Flag to not start repl"
   J no-cljs       bool            "Flag to not compile cljs"
   g guard-regexes REGEXES [regex] "Regexes controlling cljs compile"]
  (comp (watch)
        (cider/add-middleware)
        (if no-repl identity (cider/nrepl-server))
        (bs/sass)
        (reload-integrant)
        (refresh)
        (if no-cljs
          identity
          (reload))
        (if no-cljs
          identity
          (guard-task (or guard-regexes [#"clj[sc]?$"])
                      (cljs)))
        (target :dir #{"target/dev"})))
