(ns sweet-tooth.workflow.tasks
  {:boot/export-tasks true}
  (:require [adzerk
             [boot-cljs :refer [cljs]]
             [boot-reload :refer [reload]]]
            [boot
             [core :as c :refer [deftask with-pre-wrap with-post-wrap]]
             [pod :as pod]]
            [boot.task.built-in :refer [aot jar pom repl target uber watch]]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [deraen.boot-sass :as bs]
            [integrant.repl :as ir]
            [samestep.boot-refresh :refer [refresh]]))

(defn sym->var
  [sym]
  (if (symbol? sym)
    (try (let [ns-sym (-> sym namespace symbol)]
           (require ns-sym)
           (-> ns-sym find-ns (ns-resolve sym)))
         (catch java.lang.NullPointerException e
           (throw (Exception. "Could not find var " sym))))
    sym))

;; simple expire
(defn- expire-rewrite
  [index]
  (let [timestamp (quot (System/currentTimeMillis) 1000)]
    (str/replace index #"main.(css|js)" (str "main.$1?at=" timestamp))))

(deftask expire-assets
  "modify index.html to change URLs to expire assets"
  []
  (let [dir (c/tmp-dir!)]
    (with-pre-wrap fileset
      (let [index (first (c/by-path #{"index.html"} (c/input-files fileset)))
            file  (io/file dir "index.html")]
        (if (.exists (c/tmp-file index))
          (spit file (expire-rewrite (slurp (c/tmp-file index)))))
        (c/commit! (c/add-resource fileset dir))))))

(deftask build
  "Builds an uberjar"
  [v version VERSION str "version number"
   p project PROJECT sym "project name"
   m main    MAIN    sym "server ns with main fn"
   f file    FILE    str "name of jar file"]
  [version project main file]
  (c/merge-env! :resource-paths (c/get-env :source-paths))
  (comp (bs/sass)
        (cljs :optimizations :advanced
              :compiler-options {:parallel-build true})
        (expire-assets)
        (pom :project project :version version)
        (uber :exclude (conj pod/standard-jar-exclusions #".*\.html" #"license" #"LICENSE")) ; needed for arcane magic reasons
        (aot :namespace #{main})
        (jar :main main :file file :project project)
        (target :dir #{"target/build"})))

(deftask reload-integrant
  "Suspends integrant system before next task, and resumes integrant
  after. Meant to be used in conjunction with samestep/boot-refresh."
  [p prep-fn PREP-FN sym "Name of function that preps the duct config"]
  (ir/set-prep! (sym->var prep-fn))
  (comp (with-pre-wrap fileset
          (ir/suspend)
          fileset)
        (with-post-wrap _
          (ir/resume))))

(deftask dev
  [R no-repl bool "Flag to not start repl"]
  (comp (watch)
        (if no-repl identity (repl :server true))
        (bs/sass)
        (reload-integrant)
        (refresh)
        (reload)
        (cljs)
        (target :dir #{"target/dev"})))
