(ns sweet-tooth.endpoint.liberator.reitit-routes
  "Module for initializing routes created by sweet-tooth.endpoint.routes.reitit"
  (:require [duct.core :as duct]
            [integrant.core :as ig]
            [reitit.ring :as rr]
            [com.flyingmachine.liberator-unbound :as lu]
            
            [sweet-tooth.endpoint.liberator :as el]
            [sweet-tooth.endpoint.middleware :as em]
            [sweet-tooth.endpoint.routes.reitit :as err]))

;;-----------
;; duct
;;-----------
(defn liberator-resources
  [endpoint-opts]
  (let [decisions (try (-> (ns-resolve (symbol (::err/ns endpoint-opts)) 'decisions)
                           deref
                           (el/initialize-decisions (:ctx endpoint-opts)))
                       (catch Throwable t (throw (ex-info "could not find 'decisions in namespace" {:ns (::err/ns endpoint-opts)}))))]
    (->> decisions
         (lu/merge-decisions el/decision-defaults)
         (lu/resources lu/resource-groups))))

(defmethod ig/init-key ::unary-handler [_ endpoint-opts]
  (:entry (liberator-resources endpoint-opts)))
(defmethod ig/init-key ::coll-handler [_ endpoint-opts]
  (:collection (liberator-resources endpoint-opts)))

(defn endpoint-handler-key
  [endpoint-opts]
  (let [ns   (::err/ns endpoint-opts)
        type (::err/type endpoint-opts)]
    (keyword (name ns) (case type
                         ::err/unary "unary-handler"
                         ::err/coll  "coll-handler"))))

(defn add-handler-ref
  "Adds an integrant ref to an ns-route for `:handler` so that the
  handler can be initialized by the backend"
  [ns-route]
  (let [{:keys [sweet-tooth.endpoint.routes.reitit/ns handler]} (get ns-route 1)]
    (cond-> ns-route
      (and (not handler) ns) (assoc-in [1 :handler] (-> ns-route
                                                        (get 1)
                                                        endpoint-handler-key
                                                        ig/ref)))))

(defn add-middleware
  [ns-route]
  (update ns-route 1 assoc :middleware [em/wrap-merge-params]))

(defn ns-route?
  "differentiate 'normal' routes from those generated by
  ns-pair->ns-route"
  [route]
  (get-in route [1 ::err/ns]))

(defn add-ns-route-config
  [ns-route-config [_ ns-route-opts]]
  (cond-> ns-route-config
    (::err/ns ns-route-opts) (assoc (endpoint-handler-key ns-route-opts) ns-route-opts)))

(defmethod ig/init-key ::ns-routes [_ {:keys [ns-routes]}]
  (let [ns-routes (cond (vector? ns-routes) ns-routes
                        (or (keyword? ns-routes)
                            (symbol? ns-routes))
                        (do (require (symbol (namespace ns-routes)))
                            @(ns-resolve (symbol (namespace ns-routes))
                                         (symbol (name ns-routes)))))]
    (fn [config]
      (doseq [endpoint-opts (->> ns-routes
                                 (filter ns-route?)
                                 (map #(get % 1)))]
        (derive (endpoint-handler-key endpoint-opts)
                (case (::err/type endpoint-opts)
                  ::err/unary ::unary-handler
                  ::err/coll  ::coll-handler)))
      (-> config
          (duct/merge-configs
            {::router (mapv (comp add-middleware add-handler-ref) ns-routes)}
            (->> ns-routes
                 (filter ns-route?)
                 (reduce add-ns-route-config {})))
          (dissoc :duct.router/cascading)))))

(defmethod ig/init-key ::router [_ routes]
  (rr/ring-handler (rr/router routes)))

