(ns sv.ha.failover.heartbeat.core)

(defn uuid []
  (java.util.UUID/randomUUID))

(defn heartbeat [value]
  (assert (integer? (:clock value)))
  (update
   value
   :clock
   inc))

(defn take-leadership [value node-uuid]
  (-> {:leader node-uuid
       :since (java.util.Date.)
       :clock 0}))

(defn progress [node-uuid last-value current-value]
  (assert (instance? java.util.UUID node-uuid))
  (if (= (:leader current-value) node-uuid)
    ;; this instance is the leader do a heartbeat
    (heartbeat current-value)
    (let [leader (:leader current-value)]
      (if (or (nil? leader)
              ;; leader didn't do a heartbeat, it may be
              ;; unavailable.
              (= (:clock last-value)
                 (:clock current-value)))
        ;; try to take the leadership
        (take-leadership current-value node-uuid)
        current-value))))

(defn process [heartbeat-interval local-value current-value]
  (let [node-uuid (:node-uuid local-value)
        next-value (progress
                    node-uuid
                    (:last-value local-value)
                    current-value)]
    {:next-value next-value
     :sleep-duration (if (:leader current-value node-uuid)
                       ;; this machine is the leader
                       heartbeat-interval
                       ;; this machine is not the leader
                       (* heartbeat-interval 2))}))

(defn do-process [heartbeat-interval local-state state-fn]
  (let [local-value @local-state
        current-value (state-fn)
        {:keys [next-value sleep-duration]} (process
                                             heartbeat-interval
                                             local-value
                                             current-value)]
    (when (not= current-value next-value)
      (state-fn next-value))
    (swap! local-state assoc :last-value current-value)
    (Thread/sleep sleep-duration)))

(defn new-local-value []
  {:node-uuid (uuid)})

(defn start [heartbeat-interval local-state state-fn exception-handler]
  (let [first-local-value (new-local-value)]
    (reset! local-state first-local-value)
    (future
      (while (let [local-value @local-state]
               (and (not (true? (:stop local-value)))
                    ;; local-state is also used by some other process
                    (= (:node-uuid first-local-value)
                       (:node-uuid local-value))))
        (try
          (do-process heartbeat-interval local-state state-fn)
          (catch Exception e
            (exception-handler (ex-info
                                "do-process failed"
                                {}
                                e))))))
    true))

(defn stop [local-state]
  (swap! local-state assoc :stop true))

(defn is-leader? [state-fn node-uuid]
  (let [current-value (state-fn)]
    (= (:leader current-value) node-uuid)))
