(ns sv.ha.failover.heartbeat.core)

(defn uuid []
  (java.util.UUID/randomUUID))

(defn instance-ips []
  (vec
   (distinct
    (mapcat
     (fn [network-interface]
       (map
        (fn [inet-address]
          (.getHostAddress inet-address))
        (enumeration-seq (.getInetAddresses network-interface))))
     (enumeration-seq (java.net.NetworkInterface/getNetworkInterfaces))))))

(defn register [value node-uuid]
  (assoc-in
   value
   [:nodes node-uuid]
   {:ips (instance-ips)}))

(defn deregister [value node-uuid]
  (update
   value
   :nodes
   dissoc
   node-uuid))

(defn heartbeat [value]
  (assert (integer? (:clock value)))
  (update
   value
   :clock
   inc))

(defn take-leadership [value node-uuid]
  (-> {:leader node-uuid
       :clock 0}
      (register node-uuid)))

(defn progress [node-uuid last-value current-value]
  (assert (instance? java.util.UUID node-uuid))
  (if (= (:leader current-value) node-uuid)
    ;; this instance is the leader do a heartbeat
    (heartbeat current-value)
    (let [leader (:leader current-value)]
      (if (or (nil? leader)
              ;; leader didn't do a heartbeat, it may be
              ;; unavailable.
              (= (:clock last-value)
                 (:clock current-value))
              ;; leader has been deregistered
              (not (contains?
                    (:nodes current-value)
                    leader)))
        ;; try to take the leadership
        (take-leadership current-value node-uuid)
        ;; make sure this instance is register as node
        (if-not (contains? (:nodes current-value) node-uuid)
          (register current-value node-uuid)
          ;; return current-value as next last-value
          current-value)))))

;; TODO: leader could empty :nodes map every (mod (:clock value) 100)
;; or similar to garbage collect old nodes. At the moment the :nodes
;; map is emptied on a leadership change.

(defn process [heartbeat-interval local-value current-value]
  (let [node-uuid (:node-uuid local-value)
        next-value (progress
                    node-uuid
                    (:last-value local-value)
                    current-value)]
    {:next-value next-value
     :sleep-duration (if (:leader current-value node-uuid)
                       ;; this machine is the leader
                       heartbeat-interval
                       ;; this machine is not the leader
                       (* heartbeat-interval 2))}))

(defn do-process [heartbeat-interval local-state state-fn]
  (let [local-value @local-state
        current-value (state-fn)
        {:keys [next-value sleep-duration]} (process
                                             heartbeat-interval
                                             local-value
                                             current-value)]
    (when (not= current-value next-value)
      (state-fn next-value))
    (swap! local-state assoc :last-value current-value)
    (Thread/sleep sleep-duration)))

(defn new-local-value []
  {:node-uuid (uuid)})

(defn start [heartbeat-interval local-state state-fn exception-handler]
  (let [first-local-value (new-local-value)]
    (reset! local-state first-local-value)
    (future
      (while (let [local-value @local-state]
               (and (not (true? (:stop local-value)))
                    ;; local-state is also used by some other process
                    (= (:node-uuid first-local-value)
                       (:node-uuid local-value))))
        (try
          (do-process heartbeat-interval local-state state-fn)
          (catch Exception e
            (exception-handler (ex-info
                                "do-process failed"
                                {}
                                e)))))
      (state-fn (deregister (state-fn) (:node-uuid first-local-value))))
    true))

(defn stop [local-state]
  (swap! local-state assoc :stop true))

(defn is-leader? [state-fn node-uuid]
  (let [current-value (state-fn)]
    (= (:leader current-value) node-uuid)))
