(ns leiningen.deb
  (:use [selmer.parser])
  (:require [clojure.java.io :refer [copy file resource]]
            [clojure.string :as string]
            [clojure.java.shell :refer [sh with-sh-dir]]
            [clojure.set :refer [union]]
            [leiningen.ring.uberjar :as uj]
            [leiningen.ring.jar :as j]
            [leiningen.clean :refer [delete-file-recursively]]
            [leiningen.jar :refer [get-jar-filename]])
  (:import java.io.File))

(set-resource-path! (resource "templates"))


(defn deb-path
  "Returns the project directory on Debian systems."
  [project & args]
  (->> (string/replace (str (apply file "usr" "share" (:name project) args))
                (re-pattern (str "^" File/separator)) "")
       (file (:target-path project) "DEBIAN") str))

(defn deb-target-dir
  [project]
  (str (file (:target-path project) "DEBIAN")))


(defn clean
  [project]
  (delete-file-recursively (deb-target-dir project) true))


(defn deb-target-uberjar
  "Returns the filename of the uberjar in the target debian directory."
  [project]
  (str (file (deb-path project "lib") (str (string/replace (:name project) ".*/" "") ".jar"))))

(defn copy-uberjar
  [project]
  (.mkdirs (file (deb-path project "lib")))
  (copy (file (get-jar-filename project :uberjar))
        (file (deb-target-uberjar project))))

(defn sh! [cmd & args]
  (let [result (apply sh cmd args)]
    (when-not (string/blank? (:err result))
      (print (:err result))
      (flush))
    (if-not (= (:exit result) 0)
      (throw (Exception. (:err result))))
    result))

(defn mk-control
  [project]
  (let [f (str (deb-target-dir project) "/" "DEBIAN/control")]
    (.mkdirs (.getParentFile (file f)))
    (spit f (render-file "control.tmpl" project))))

(defn mk-init
  [project]
  (let [f (str (deb-target-dir project) "/" "etc/init/" (:name project) ".conf")]
    (.mkdirs (.getParentFile (file f)))
    (spit f (render-file "init-conf.tmpl" (assoc project :jar (str (string/replace (:name project) ".*/" "") ".jar"))))))

(defn conffiles
  [project]
  (let [f (str (deb-target-dir project) "/" "DEBIAN/conffiles")]
    (spit f (str "/etc/init/" (:name project) ".conf\n"))))

(defn maintainer-scripts
  [project]
  (let [postinst (str (deb-target-dir project) "/" "DEBIAN/postinst")
        postrm (str (deb-target-dir project) "/" "DEBIAN/postrm")]
    (spit postinst (render-file "post-inst.tmpl" (assoc (:deb project) :name (:name project))))
    (spit postrm (render-file "post-rm.tmpl" (assoc (:deb project) :name (:name project))))
    (sh! "chmod" "755" postinst postrm)))

(defn mk-deb
  [project]
  (with-sh-dir (:root project)
    (sh! "fakeroot" "dpkg-deb" "--build" (deb-target-dir project))))

(defn mv-deb
  [project]
  (with-sh-dir (:root project)
    (sh! "mv" "target/DEBIAN.deb" (str "target/" (:name project) ".deb"))))

(defn ring
  [project]
  (doto project
    (clean)
    (uj/uberjar)
    (copy-uberjar)
    (mk-control)
    (mk-init)
    (conffiles)
    (maintainer-scripts)
    (mk-deb)
    (mv-deb)))

(defn deb
  "I don't do a lot."
  [project & [command]]
  (case command
    "ring" (ring project)
    (println "what?"))
  )
