(ns common.http
  "HTTP client helpers.

   Note: The Ring spec [1] asks for lower-cased strings for HTTP header names.
   We respect that convention here even though an more flexible implementation
   like clj-http's [2] would ultimately be preferable.

   [1] https://github.com/ring-clojure/ring/blob/master/SPEC
   [2] https://github.com/dakrone/clj-http#headers

   >> ctdean jimbru"
  (:refer-clojure :rename {get core-get})
  (:require
    [cheshire.core :as json]
    [clojure.set :refer [intersection subset? rename-keys]]
    [clojure.tools.logging :as log]
    [common.coll :refer [keys-set]]
    [common.string :refer [starts-with?]]
    [org.httpkit.client :as http])
  (:import com.fasterxml.jackson.core.JsonParseException))

(defn- response->json
  "JSON-decodes HTTP response bodies. Transforms JSON string keys into Clojure keywords
   for easier access. Note that we are NOT lazily decoding so that we avoid deferring
   exceptions (which can't be caught inside this function)."
  [response]
  (try
    {:status :ok
     :body (json/parse-string-strict (:body response) true)}
    (catch JsonParseException e
      {:status :error
       :error ::error-response-json})))

(defn- decode-response-body
  [response]
  (let [content-type (:content-type (:headers response))
        decoded (when (starts-with? content-type "application/json")
                  (response->json response))]
    (condp = (:status decoded)
      :ok    (assoc response :body (:body decoded))
      :error (assoc response :error (:error decoded))
      nil    (do
               (log/debug "Unknown Content-Type, skipping decode:" response)
               response))))

(defn- process-response
  [response]
  (cond
    (:error response)
      (-> response
          (rename-keys {:error :debug})
          (assoc :error ::error-network))
    (not (<= 200 (:status response) 299))
      (assoc response :error ::error-response)
    :else
      (decode-response-body response)))

(defn- opt-json-body [opts]
  (if-let [json-body (:json-body opts)]
    (do
      (when (get-in opts [:headers "content-type"])
        (log/warnf "Overwriting existing Content-Type header in request: %s" opts))
      (-> opts
          (assoc :body (json/generate-string json-body))
          (assoc-in [:headers "content-type"] "application/json; charset=utf-8")
          (dissoc :json-body)))
    opts))

(defn- opt-jwt [opts]
  (if-let [jwt (:jwt opts)]
    (do
      (when (get-in opts [:headers "authorization"])
        (log/warnf "Overwriting existing Authorization header in request: %s" opts))
      (-> opts
          (assoc-in [:headers "authorization"] (str "JWT " jwt))
          (dissoc :jwt)))
    opts))

(defn request
  "Performs an HTTP request. Mirrors calling semantics of http-kit; including a callback
   function argument will run the request async. Otherwise a promise is returned.
   Supports the following options:

    :url            string
        URL to request.
    :method         keyword
        HTTP method.
    :headers        {string string}
        Additional HTTP headers to include. Note that some headers
        will be included automatically.
    :query-params   {string string}
        Will be URL-encoded and added to the URL's query string.
    :form-params    {string string}
        Will be form-encoded and set as the request body.
    :body           string
        Sets the request body.
    :json-body      {any any}
        Will be JSON-encoded and set as the request body.
    :jwt            string
        Sets a JWT authorization header with this value.
    :timeout        int
        Sets the timeout in milliseconds.
    :keepalive      int
        Keep alive idle time in ms. Use -1 to disable.
    :user-agent     string
        Sets the User Agent
  "
  ([opts]
    (let [p (promise)]
      (request opts #(deliver p %))
      p))
  ([opts f]
    (assert (subset? (keys-set opts)
                     #{:url :method :headers :query-params :form-params
                       :body :json-body :jwt :timeout :keepalive :user-agent})
            "Unsupported option.")
    (assert (intersection (keys-set opts) #{:form-params :body :json-body})
            "Options :form-params, :body, and :json-body are mutually exclusive.")
    (http/request
      (-> opts opt-json-body opt-jwt)
      #(-> % process-response f))))

(defn- convenient-request [& opts]
  (request (apply hash-map opts)))

(defn get
  "Convenience function for GETs."
  [url & opts]
  (apply convenient-request :method :get :url url opts))

(defn post
  "Convenience function for POSTs."
  [url body & opts]
  (apply convenient-request :method :post :url url :json-body body opts))

(defn put
  "Convenience function for PUTs."
  [url body & opts]
  (apply convenient-request :method :put :url url :json-body body opts))
