(ns common.db.load
  "Installs database fixture data. Provides an alternative -main so that it can
  be run from the command line: `lein run -m common.db.load`.

  The initial data file should be in a format that looks like this:

      [
        :table       ;; Table name as keyword
        [            ;; Row data
          {:id 1
           :foo \"bar\"}
          ...
        ]
        :another-table
        [
          {:baz :quux}
          ...
        ]
        ...
      ]

  This library also provides the following custom EDN tag/converters:

    - #date \"yyyy-mm-dd\"
      Yields a SQL-compatible Date object.
    - #datetime \"yyyy-mm-ddThh:mm:ssZ\"
      Yields a SQL-compatible DateTime object.

  >> jimbru"
  (:require
    [clojure.edn :as edn]
    [clojure.java.jdbc :as jdbc]
    [clojure.java.io :as io]
    [common.convert :as convert]
    [common.db.util :as util]
    [conf.core :as conf])
  (:gen-class))

(def dbspec-conf {:connection-uri (conf/get :database-url)})

(def ^:private edn-readers
  {'date     util/str->sql-date
   'datetime util/str->sql-date-time
   'uuid     convert/str->uuid})

(defn- read-edn-file [filename]
  (->> filename
       io/resource
       slurp
       (edn/read-string {:readers edn-readers})))

(defn- read-data-file [filename]
  (let [extension (last (clojure.string/split filename #"\."))]
    (case extension
      "edn" (read-edn-file filename)
      "clj" @(load-file filename))))

(defn load-data-file
  [file dbspec]
  (let [filename  (or file "initial_data.edn")
        file-data (read-data-file filename)]
    (jdbc/with-db-transaction [tcon dbspec]
      (doseq [[table rows] (partition 2 file-data)]
        (apply jdbc/insert! tcon (name table) rows)))))

(defn -main
  [& args]
  (let [filename (first args)
        dburi (second args)]
    (common.db.load/load-data-file filename (if dburi
                                              {:connection-uri dburi}
                                              dbspec-conf))
    (println "Completed loading fixtures")))
