(ns ring.adapter.finagle
  "Adapter for the Finagle RPC system"
  (:import com.twitter.finagle.Service
           com.twitter.util.Future
           java.net.InetSocketAddress
           com.twitter.finagle.builder.ServerBuilder
           (com.twitter.finagle.http Request Http)
           clojure.lang.Reflector)
  (:require [ring.util.finagle :as finagle]))

(defn- create-service
  "Returns a Finagle Service for the given Ring handler"
  [handler]
  (proxy [com.twitter.finagle.Service] []
    (apply [request]
           (let [request-map (finagle/build-request-map (Request/apply request))
                 response-map (handler request-map)]
             (Future/value (finagle/build-response response-map))))))

(def valid-options
  [:codec
   :reportTo
   :name
   :bindTo
   :logger

   :maxConcurrentRequests
   :hostConnectionMaxIdleTime
   :hostConnectionMaxLifeTime
   :requestTimeout
   :readTimeout
   :writeCompletionTimeout

   :sendBufferSize
   :recvBufferSize

   :tls
   :startTls
   :channelFactory
   :traceReceiver])

(def default-options
  {:codec (Http/get)
   :name  "RingServer"})

(defn- build [options]
  (reduce (fn [builder opt]
            (if (contains? options opt)
              (Reflector/invokeInstanceMethod builder (name opt) (into-array Object [(opt options)]))
              builder))
          (ServerBuilder/get)
          valid-options))

(defn- create-builder
  "Construct a Finagle builder"
  [options]
  (let [bindTo (InetSocketAddress. (:host options "localhost") (:port options 80))]
    (build (merge {:bindTo bindTo} default-options options))))

(defn run-finagle
  "Server the given handler according to the options.
  Options:
    :port
    :host
    :name"
  [handler options]
  (.unsafeBuild (create-builder options) (create-service handler)))
