(ns splendid.jfx
  (:import (splendid SplendidJFX)
           (java.net MalformedURLException URL)
           (javafx.application Platform)
           (javafx.beans.property BooleanProperty DoubleProperty
                                  FloatProperty IntegerProperty
                                  LongProperty ObjectProperty StringProperty)
           javafx.beans.binding.Bindings
           (javafx.event EventHandler)
           javafx.fxml.FXMLLoader
           (javafx.scene Group Node Parent Scene)
           javafx.scene.control.Accordion
           (javafx.scene.layout FlowPane GridPane HBox Pane VBox)
           (javafx.stage Stage)))

(defonce ^Stage primary-stage
  (when-not *compile-files*
    (let [s (promise)]
      (SplendidJFX/launchApplication #(deliver s %))
      @s)))

(defmacro jfx
  "Runs `body` on the JavaFX Application Thread and blocks until execution has
  finished and returns its result."
  [& body]
  `(let [f# (fn [] ~@body)
         p# (promise)]
     (if (Platform/isFxApplicationThread)
       (f#)
       (do
         (Platform/runLater #(deliver p# (f#)))
         @p#))))

(defprotocol Showable
  "Objects that can be shown on a javafx.stage.Stage."
  (show* [obj stage]
    "Displays `obj` on the given Stage `stage`."))

(extend-protocol Showable
  clojure.lang.Fn
  (show* [f stage]
    (jfx (f stage)))
  Parent
  (show* [p, ^Stage stage]
    (jfx
     (.setScene stage (Scene. p))
     (.show stage)))
  java.util.Map
  (show* [m stage]
    (show* (get m :ui) stage)) ; calling show* with the Parent `(:ui m)`
  )

(defn show
  "Takes a Showable object `s` and displays it on the Stage `stage`.
  If no Stage is given, the `primary-stage` is used by default."
  ([s] (show* s primary-stage))
  ([stage, s] (show* s stage)))

(defn jfx-class? [^Class c]
  (cond
   (symbol? c) (jfx-class? (resolve c))
   (class? c)  (boolean (.startsWith (.getName c) "javafx."))
   :else false))

(defn jfx-instance? [obj]
  (jfx-class? (class obj)))


;; ## FXML

(defn scenegraph
  "Takes a Node `x`, typically a layout (something that can have children),
  and returns the scenegraph from this point on, including `x`,
  represented as a lazy seq."
  [^Node x]
  (lazy-cat
   [x]
   (when (instance? Parent x)
     (map scenegraph (.getChildrenUnmodifiable ^Parent x)))))

(defn collect-ids
  "Takes a seq of `nodes` (i.e. a flattended scenegraph, see `flatten-scenegraph`)
  and returns a map of node ID to node. For example: a Button `b` with the ID
  `ok-bt` would be returned as `{:ok-bt b}`."
  [nodes]
  (reduce #(let [id (keyword (.getId ^Node %2))]
             (cond
              (nil? id) %1 ; the control has no id, so we ignore it

              (contains? %1 id) ; ==>
              (throw
               (Exception.
                (format "ID `%s` was assigned at least twice to a control." id)))

              :else (assoc %1 id %2) ; keep this id/control pair
              ))
          (sorted-map)
          nodes))

(defn- graph->map
  "Takes a `graph` of JFX nodes and returns a map of those components, which
  were assigned an ID. The IDs become Keywords, and the values in the result map
  are the corresponding instances. The toplevel container `graph` is returned
  under the key `:ui`, even if it wasn’t assigned such an ID. If the toplevel
  component was assigned an ID, then also that will be member of the result map.
  Private code. Don’t call."
  [graph]
  (-> graph
      scenegraph
      flatten
      collect-ids
      (assoc :ui graph)))

(defn load-fxml
  "Loads the FXML file `f` and returns a hashmap `h`, containing the top-level
  node of `f` (typically a layout with its components), mapped to the key `:ui`.
  `h` also contains all components that were assigned an ID in `f`.
  You may not use the ID `\"ui\"` in your FXML files if you want to load them
  via `load-fxml`.

  Optionally takes a ResourceBundle `rb`. If your FXML file contains text
  contents prefixed by a `%` you must provide a resource, as otherwise the file
  can’t be loaded."
  ([f]
     (graph->map (FXMLLoader/load (try (URL. (str f))
                                       (catch MalformedURLException e
                                         (URL. (str "file:////" f)))))))
  ([f rb]
     (graph->map (FXMLLoader/load (try (URL. (str f))
                                      (catch MalformedURLException e
                                        (URL. (str "file:////" f))))
                                  rb))))

;; Okay, this section will provide some macro magic and eval calls.
;; In general eval is perceived as evil, and this feature may
;; get removed in future versions.
;; But for now I see this experimental feature as beneficial,
;; so here it is.

(defonce ^{:doc "Function cache, maps from fxml to fn."}
  fncache (atom {}))

;; Constructs the let, mostly via the mapcat, which produces key value pairs.
;; The mapcat fn gets the keys `k` (Keywords) and values `v` (JFX components)
;; from the `merge` of the loaded FXML `files`. Out of the keys it produces
;; symbols. The values can not simply get inserted into the let, as those are
;; instantiated JFX objects, and Clojure doesn allow expressions like
;; (let [pane #<VBox VBox@11db68d>] …).
;; So instead it refers to the binding `fxml`, which is the first thing that
;; gets bound in the let block.
;; The mapcat fn does not simply output [(symbol (name k)) (k fxml)], but
;; instead it will typehint the symbol first, so that in the body no reflection
;; will occur.
(defn construct-fxml-let
  "Constructs a `let` with components from FXML `files` with the given `body`.
  Private code. Don’t call."
  [files body]
  `(fn []
     (let [~'fxml (apply merge (map load-fxml ~files))
           ~@(mapcat (fn [[k v]]
                       [(with-meta (symbol (name k))
                                   {:tag (symbol (.getName (class v)))})
                        `(~k ~'fxml)])
                     (apply merge (map load-fxml files)))]
       ~@body)))

(defn construct-cached-fxml-let
  "Similar to `construct-fxml-let`, but caches the generated FN.
  Private code. Don’t call."
  [files body]
  `(swap! fncache assoc ~files
          ~(construct-fxml-let files body)))

(defmacro with-fxml
  "Loads and merges the specified FXML `files` in the given order, and makes
  all components that were assigned an ID available in the `body`, type-hinted.
  Calls `load-fxml`, so make sure to specify your topmost container last, to
  have it available as `ui` in the `body`.
  Inside the `body` you can refer to `fxml`, which provides a map of all merged
  components from the `files`, including the last toplevel container, under the
  key `:ui`.

  `with-fxml` expands into code which will generate and compile code at runtime.
  It will thus potentially delay the execution of your code by 3-8 msecs.

  Warning: this is an experimental feature, and it is not certain that it will
  make it into the 1.0.0 release."
  [files & body]
  `((eval (construct-fxml-let ~files '~body))))

(defmacro with-fxml*
  "Like `with-fxml`, but memoizes the specified `files`.
  If this block of code is executed again it will call the memoized function
  and thus save 20-30 msecs in later calls."
  [files & body]
  `(if (contains? @fncache ~files)
     ((get @fncache ~files))
     ((get (eval (construct-cached-fxml-let ~files '~body))
           ~files))))


;; ## Containers

(defprotocol Container
  (add [pane elements] [pane elements obj]
   "Adds elements to the container. Optionally takes an obj which may be used
  by specific Containers (i.e. MigLayout)."))

(defn- add-template
  "Serves as a template for implementing the `add` function.
  Private code. Don’t call."
  []
  `(~'add [~'pane ~'elements]
     (if (jfx-instance? ~'elements)
       (.add (.getChildren ~'pane) ~'elements)
       (.addAll (.getChildren ~'pane) (to-array ~'elements)))))

(extend-protocol Container
  Pane #=(add-template) ; expanding `default-add` during read time
  GridPane
  (add
    ([pane elements]
       (if (jfx-instance? elements)
         (.add (.getChildren pane) elements)
         (.addAll (.getChildren pane) (to-array elements))))
    ([pane element [column row colspan rowspan]]
       (let [colspan (or colspan 1)
             rowspan (or rowspan 1)]
         (.add pane element column row colspan rowspan))))
  Group #=(add-template)
  Scene (add [scene root-pane] (.setRoot scene root-pane))
  Stage (add [stage scene] (.setScene stage scene))
  Accordion
  (add [accordion elements]
    (if (instance? Node elements)
      (.add (.getPanes accordion) elements)
      (.addAll (.getPanes accordion) (to-array elements))))
  )


;; ## Events

(defmacro defhandler
  "Takes an `event` which denotes the event for which you want to implement
  a handler for a given `node`."
  [event ^Node node & body]
  (let [f (symbol (str ".setOn" (subs (name event) 2)))]
    `(~f ~node
         (reify EventHandler
                (handle [this# event#] ~@body)))))

(defmacro defhandler*
  "Like `defhandler`, but takes `fn` of two arguments: this and the event."
  [event ^Node node fn]
  (let [f (symbol (str ".setOn" (subs (name event) 2)))]
    `(~f ~node
         (reify EventHandler
           (handle [this# event#] (~fn this# event#))))))


;; ## Bindings

(defprotocol Binding
  (bind [property properties f]
    "Creates a binding for `property` which observes changes in one or more
  `properties` and runs the zero-arity function `f` when at least one of the
  `properties` changed.
  The return value of `f` will be the new value of `property`."))

(defn- binding-template
  "Template that serves for nearly identical code, to implement bindings.
  Private code, don’t call."
  [name]
  (let [method-name (symbol (str "Bindings/create" name "Binding"))
        property-name (symbol (str name "Property"))]
    `(~'bind [~'p ~'observables ~'f]
       (.bind ~'p (~method-name ~'f (into-array ~property-name  ~'observables))))))

(extend-protocol Binding
  BooleanProperty #=(binding-template Boolean)
  FloatProperty   #=(binding-template Float)
  DoubleProperty  #=(binding-template Double)
  IntegerProperty #=(binding-template Integer)
  LongProperty    #=(binding-template Long)
  StringProperty  #=(binding-template String)
  ObjectProperty  #=(binding-template Object))

(defmacro defbinding
  "Like `bind`, only that you can provide a body `expr` instead of a function."
  [property properties & expr]
  `(bind ~property ~properties (fn [] ~@expr)))
